"""
Contains embedded image and icon resources. Auto-generated.

------------------------------------------------------------------------------
This file is part of h3sed - Heroes3 Savegame Editor.
Released under the MIT License.

@created     21.03.2020
@modified    11.06.2024
------------------------------------------------------------------------------
"""
try:
    import wx
    from wx.lib.embeddedimage import PyEmbeddedImage
except ImportError:
    class PyEmbeddedImage(object):
        """Data stand-in for wx.lib.embeddedimage.PyEmbeddedImage."""
        def __init__(self, data):
            self.data = data


"""Returns the application icon bundle, for several sizes and colour depths."""
def get_appicons():
    icons = wx.IconBundle()
    [icons.AddIcon(i.Icon) for i in [
        Icon_16x16_32bit, Icon_16x16_16bit,
        Icon_24x24_32bit, Icon_24x24_16bit,
        Icon_32x32_32bit, Icon_32x32_16bit
    ]]
    return icons


"""Heroes3 Savegame Editor application 16x16 icon, 32-bit colour."""
Icon_16x16_32bit = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAALHRFWHRDcmVhdGlvbiBUaW1l"
    "AEwgMjEgbeRydHMgMjAyMCAxNTo0NDo0NyArMDIwMMsTHzoAAAAHdElNRQfkAxUNODRflfVU"
    "AAAACXBIWXMAAAsSAAALEgHS3X78AAAABGdBTUEAALGPC/xhBQAAAtxJREFUeNp1U2tIk2EU"
    "fr5vn9vUecn7baQkS8s0MlNL8dMkZPnDbJSIUBb+iJAIuuAPtxUEhRKFPyT8UxIhIUKii7Tc"
    "N9HhpRAvlWWbl2mpqVOXc6nbW5tm0+zAezgv5/Y857wvsFUUv0+a3ZDL5Zv2dt//RNHy9CR5"
    "UJJIkmIota4tnrx5VUs2khStz7JJhTyTbC9CO1862zshzUpC5B6GndAP4kjcAmQymdLh03Yg"
    "NTnUbrI7to/0h7rm/kGSIaFIc7U7UVW5El37YQeK3PRQ0liZbO+u3p7HsytpRrT6Yk4gmyUr"
    "hjc9Cl+fWUyNrcBbNIP4tNuI9NVjfLAHFqFk5Iv+2xPnAoxd+XsZ2Q9dUwgNeoQfi/MYGliF"
    "xQLoxwjExkoYDMCYbhaM2e0f+I4ZWK2rWBW6IkAcjmBxACZGbDAt2UDWXLC6MA1GtBeSQ/tw"
    "PicIIR5baTgKzC0buaIrx9HbrcfQx6/o7wN8PBiYFy3gudIIDGAQFBaO09e1ytiIzZX+paBq"
    "tqXTJg2RJi1ht9gH48HAvNGKxcWNIDeCntfduHftqPJmuZbaaY2Kq5clyLlwB7tCQzA2x8PA"
    "ZwF4PL7DOTtJYYX2RF6eFLJc6RYKzB/jReMM+nqeo66+lwsTUazY24YQiYvDx3e3op4bBdd6"
    "F/GpNrZ2/TFptkyT3U+Rs9G0fdeK2jIBKZVRhHvsRlZMn0jhMZrcKHAn/kKoi064keoKX1LM"
    "rr9IB4VT2Snq5HAGS8RmXxM3aqQRFcuHZ2AU3nX149ylAmg1Zu67BelVTWblsm4BP9dodrOA"
    "69owOz66xjUMOmBpuntW4BVAwc83BhPvB6BS1aLNQJQbYDmDKQbT5o0hyjJoRXycD4YnNwNQ"
    "89KqFAkIgqJL0dFUhpa3Zs6Js8ZE8RHhx6wjCLbZOHrBiMSEMM5pJLce1iVw+YUlXHnDEpvC"
    "sqzzLxzSmzjBgTNsfqZQ8Qs1XyOB+yzbpwAAAABJRU5ErkJggg=="
)


"""Heroes3 Savegame Editor application 16x16 icon, 16-bit colour."""
Icon_16x16_16bit = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAMAAAAoLQ9TAAAALHRFWHRDcmVhdGlvbiBUaW1l"
    "AEwgMjEgbeRydHMgMjAyMCAxNTo0NDo0NyArMDIwMMsTHzoAAAAHdElNRQfkAxUNOQKJNFGM"
    "AAAACXBIWXMAAAsSAAALEgHS3X78AAAABGdBTUEAALGPC/xhBQAAAaRQTFRF////j2gcc0kc"
    "UU08VFRUcGZASkIpeHNcjXctAAAAwJ8H+vPbdEwLNCYAb2Q2cUYGWkgbXSwBWEkDtJgI4spe"
    "rJKAdloKfGMwUUw8eUsFjXoJpYIHzrcyx72swKRePB0BmXAEZjwCQCsBVykCrIsHfVQBwKAK"
    "+vPZYTsCdlwLFQgAZUYKopMTblggj3IyPSQAWTABfEoDqowI7Nxev6+BQzAJUTIMhG8KvaM0"
    "5tKIvZ8vUzAFXzoJkGYIuaSEkXAo4cyEVFJKTDgOlnADrIQVcUICi2ACjWMCzLyDl24aiXQ6"
    "SkUC0LxNnHsxjWcaVikCm3EFpH8DyLKEw6VPSC4AXE4Ta00Ky7BTspA0g1cCiWQBro0L6dNc"
    "sZyAq4kvrYs6Gg0AX1EYcEUBtZMzmXMenncElm8Dr5EG7uJyrZBZhVcFTTsimnsr6diapYEu"
    "w6MJpYED0a8Hyq1Zo3oYSUUyJBUAqI4wQiYBu6ILtpsJ3cU0vbmsUkEbtZc5h2kU2sFvXE8F"
    "wqIL8+idi2wU0LdfvqRLkHk3OCwB3cQzwqZMk35Efn5+nz2PAgAAAAF0Uk5TAEDm2GYAAADS"
    "SURBVHjaY2AAg24g5mSAAs7WtvaOagbOzi6YSF19Q2MTC0NzC5RfWlZeUVlVXVML1cOWk5uX"
    "X1DIXVRcAhFITEpOSU1Lz8jMyoYIhIaFR0RGRcfExsUngAW8fXz9/AMCjYOCGUIgSpycXVzd"
    "3I09PL2gtlhYWlnb2Brb2Ts4Qtylq6dvYGhkbGJqZg5RoaTMoKKqpq6hqaWtA+JLSDIwSEnL"
    "yMrJMzAoKAIFBAQZGISERUTFgGxxkAoOkDlc3Dy8QJqPH0gwMgEJZhYGVpA32BkA2WYnWed2"
    "jJAAAAAASUVORK5CYII="
)


"""Heroes3 Savegame Editor application 24x24 icon, 32-bit colour."""
Icon_24x24_32bit = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAYAAADgdz34AAAALHRFWHRDcmVhdGlvbiBUaW1l"
    "AEwgMjEgbeRydHMgMjAyMCAxNTo0NDo0NyArMDIwMMsTHzoAAAAHdElNRQfkAxUNOA0AkH1c"
    "AAAACXBIWXMAAAsSAAALEgHS3X78AAAABGdBTUEAALGPC/xhBQAABctJREFUeNqNVXtMU2cU"
    "//W2pa19QhHKQ1oUQZyPoqgwmS3TqKhb2B9sI0PBuWwLmgwS/9hD45ZMZ7Zk6ozZI3GW+Zwx"
    "C0QBnQLqfCEQisBwFkoBRQotLY9CS3vvt1tUqIrGLzm5+c73+87v3nN+51zg+aVmTcfawieO"
    "tLQ03c6dO/0++TNY3WOT4xWXOj0lpOrDrBiyOTu1nd3rWcvduzONVFw4Q5KSko4EYHM/yp5N"
    "8jdpyYLEqCOvSqDbvnUdOfadlhTtmkuyN8xs/ziTcrSUy4mjYwepu32FsJh9j7FVzRUF5OfC"
    "CHLyYFag/6lFPet40NGIVlM7klKXQDN7sSYuga/ovk8w9LAW8xOVSE1N1T9OI+prL0Eg4WLp"
    "svFsal+JwG7rh1wuxtFjN2C31iI6ggdGQMNhv4qRvjLs2bPHH8iwIIqjcVjtaGrz4Pt9x/1X"
    "DVMRcAM3KhmK16fKVPGJMbhc3oxlWhfUGi68HhpuFwOxvBNR0ZnoddCadWlSRXJSDM7f6MbR"
    "ko4C9vqvLyNQL54fffnHL3XaBfNDsWh5FpRSK7wuOzSRY+ju8kESxINUbkPYjC1IT1EjROJA"
    "XVUJ3CQYvQP8887BkeqXpShv26ZE7enfS8ETSkHTNJQRCfARBm1mCqOjgM3BoNvCgcd9BRyK"
    "hwunyllCEXKzV0AVrtj/ItXwJnNvgoh9VpRVQB11G7SPhlDABu6gwXA57Jv4MOjmY2SoGPKw"
    "DITFROHOzTrUV5ei2zz6QllOFJlhfLjv5iFxvhqL0jIQN28Rm3cvHvQQDIzS4BAeSxgEetQD"
    "Hk+IYY8AoZEy5BSuQuG7oQgWov6JuqYkkIe68E7WIhBCMEZ72cLyYWoEeh6wn0k4GBrwwWlz"
    "gxljxvFSoRCrslbi3n0xBj1ewyzFuEwNz5I8ITD8dtJpXJmhhbO/C38XV6Lm1jV4QTDsAmJj"
    "RRDL2Py7aPCkj3RBQGOwtwfHi4qx60iPIWEWCl6fN975mqlq0FFnJElvvfcHObBDimFbN2ao"
    "w+HuZYP5GNxtdiM4hGCEP9k2KjWF2n9qkLMhmX0R0+Xj13s4rPvAyxotNzl+BJb/+IjQrMA0"
    "mQDxC8UYIxQcI1x4vCKIxBOagL2PgkylQsqatdj+yXp/qNyXFtkv1dxNr2F93ldIy1gHSUgs"
    "mkwUWloZtN9jzeSFQDwJd9p8aG50wHz3JmbFq/DLoW/9Un2OhBe4qa91wOO5jTF3G0pOXzW2"
    "m6EVshJNmMmBcjqNkEjBBFYaxkq42o2LFxvwsK0SymifYl4i9E0tKGaPB57gnhoVtQ1Dmb0N"
    "DTBe68SperyfvITK08ZS6LMRzJlHYUYMQUTcZlQbObhT+Sdaex249+8ArjV6kfO2HCnLudq2"
    "rjFVrw0lU6WoSCEDIiM47LDjjDskCj7mJnMQOYNgmpzL6obP1iED1rtNIK5O6JcGw+wARggK"
    "Dp0YgGBUgT1fKPK2pXP2PUugTkmeU79xOQ/WbmKxDzN+TTuHRxiLMCQImjh/JmlMZ7uX9YFL"
    "D2HZah2ultmdi8NJ3iiDAzVmWn/4xAOIvC62OSdH98Qsyl4drXV7KUyfxjGUmtDA+hrOlHkN"
    "JgsNVSwPMqUACuUWdHZ60WI8C6EwHOdaifHsXRRNBGMT7nGtxBvLOM8Xmc/3wNruM1I+xhBY"
    "F/cgDYpHQSLygk8xEAVxIIQVu3/6y39cEAA1ShgYDp405VktZNI7RwNd4cYwsvtTMUmbPuVv"
    "r+rSYQG5X7OR+Dx2krNESPKzZCRUgc+mwO4y7E8nBauC6vPTH40ManM6kKFbjEFahc+/yXQu"
    "V463e+AyXLyphFCpR9HRU7D3ewwxEdHYmrt0St3LZPFY80G+9s21yYZzPyjU3JmhcA71Daoc"
    "Fpu2tdGqF8FtuWMjVwLuNDzslzjLK0xrS8sqCmrMjr02l1hl7nJqu3uGvmbPrQFYS6vZqr/V"
    "3K+qrG7XXLg+VPw/NVxfuUguWa8AAAAASUVORK5CYII="
)


"""Heroes3 Savegame Editor application 24x24 icon, 16-bit colour."""
Icon_24x24_16bit = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAMAAADXqc3KAAAALHRFWHRDcmVhdGlvbiBUaW1l"
    "AEwgMjEgbeRydHMgMjAyMCAxNTo0NDo0NyArMDIwMMsTHzoAAAAHdElNRQfkAxUNOCBFTyEp"
    "AAAACXBIWXMAAAsSAAALEgHS3X78AAAABGdBTUEAALGPC/xhBQAAAtlQTFRF////g1sWbEAI"
    "bD4VXCgCIiAZPjw2MCwgNTAbMy0cPTs0MisTupxJd2Ujd2Qa7N6dnZyVSSMBiG4qf282AAAA"
    "rp5nt59wrIk0cV0WhWgLPBUAvJ0H5Mtd/PntXDUIalUOoJBBcEQGln89YiwBgmkEtZoJxKMG"
    "7diI18rAiF4PRDQHUCoBRzkjMhgAkXpEd04GincHtZYL6NaKzsS/gF4jv6JZQysBs5RKjmAC"
    "ODQogVMDUiUBVEoFr5cKnXkEpoME2MMn6+HEqY5zt5tETSQBu5wGIhgAaT0DXTMCY1IEflIC"
    "qIQGw6UM+OzGppGAimsTGhEBNSIDpYQcsqIQEQkATiICVEQCYzQCnHUE7+F8vbmsSjgDPCIB"
    "EwMAonwRno0MLCkAjHIat59hsZc+YkEAQx4AdEUDhVwB5tYy+vLZoYpEWz0Ie2EdYj4KdVoG"
    "vqYr59eX5tKMxKlCe1wSPhoCo4AJlHREb0sQwqRL4Mh7mJWGX0gAnHccqopBuJchqYEPZz4E"
    "bkkFimAJm3UJpIhImngh4s6FvaJjKyIKvJ4XkmkCrokabEIBkWUCqoc9STsYMzAB3stonX4P"
    "jGgdp34KpH8Cn35EwJ0+TjsRWzsAOiwAmYkRm4I4rZAs0bhfckMCq4oF4cosoH5HzrRWqZph"
    "TCsC1rtklXIFeEwCvJwN5M8xk3Y/il4TjHcggmMOon0Zs5cH59g49u7Rn304dkQFc0USHhIJ"
    "n4tFyK5RwqZZwZ8G8eWI7eC2iV8dzLFdODQktJZQxKZE6dWRr40yyqUEvJUI278F7N6y0Ldm"
    "l20IbVgfIB0UGA4Aqo80uqELX19fkHk9U0YVRjgLdVwlZVkjppAL1LYM8N6Kv7+/LhsBlXEZ"
    "cloVZ1kpWUwdeXFV4shI+vPUOjo6pIksnn0c3cqEdGtKpIcwoX8qWUwizcrB1b1wp5NVQTgS"
    "YlUeYVk5iH08MTExal4wLikbeN0ivQAAAAF0Uk5TAEDm2GYAAAGxSURBVHjaY2DAB968fccg"
    "8j7qA5B55eMnJImnz56/8LwW+ZKBQeTVrddIEnfXzbtnsPL+g4cMIo8ePxFBSFxKvHzl6rlr"
    "12+I3Lx1+w5C/EzxWe3l51RdFzqev1BxESHOeeDgocPNR+yPHvM9fuLkqdMwcZHtHjsKdy73"
    "tvPetXvP3rh9++E61oavW7/errfXbsPGTZu3bN0G15G4dO6yiuUr0u1WrspcvWbx5BCYzOw5"
    "c+dVzPdI71qwMHPR4iUMujD3Tpg4abJ/j/UUu6kZmdOmz2CYOQumpaOzq9uqx6u3KyPTuamv"
    "H+He6prauvqGxqbmjMyW1rZ2uLhIfkFhUXFJqHtpRmZZeUVlFcJZSckpqalp6aoZmVnZObl5"
    "CTCZEJdQhjCv8Ai1yCiT6JjYuHiYjJu7B4OnvZe3j6+flX8AQ2BQMETc0sqagcHG1s7ewdHJ"
    "2QUoYOaqDbZCTx9EGqgaGhmbmII8Z2ZuAdahpAwiVVTV1DU0tUBMbR1dECUhKQXWKC0jKycP"
    "MVxBEUTy8EJ4fPwCgkLCYDWiYuJAipGJmQXEZWVjYGDngEQdFzcDAJiOlEQRp6PjAAAAAElF"
    "TkSuQmCC"
)


"""Heroes3 Savegame Editor application 32x32 icon, 32-bit colour."""
Icon_32x32_32bit = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAALHRFWHRDcmVhdGlvbiBUaW1l"
    "AEwgMjEgbeRydHMgMjAyMCAxNTo0NDo0NyArMDIwMMsTHzoAAAAHdElNRQfkAxUNNwIXt3wC"
    "AAAACXBIWXMAAAsSAAALEgHS3X78AAAABGdBTUEAALGPC/xhBQAABqtJREFUeNrNV2lMVFcU"
    "/t4MM8wCzCCLooiyCmjFpWqL2haionHBkNgaNTEmasVStT9M49LEBlNTbaqxVEVNjFSNC9KW"
    "oqA2YhBtRSlSqbhRcFCBQWBYZmGZub33jjMwMjPAv57k5b133zv3nvOd75x7LvA/FtLnchhn"
    "4mR8ID2n4uFqklvnZ/GHkHELMWbSdmIhRGDjty+qYay/xKyAIAhsEaGvXkleMroNTW/rDd17"
    "Q10e2bEcZP8aMSn7LYV7dOEbCam8CKIpl/HvTpAghpdnyO7lIn7Vlu8ZEAmRqw8VpcfQqhcQ"
    "vzAJ0TM28DHvIAmqXwZA/8wEQSx1rld2DkZYkLAkHnL1hAE9dWlAVmYuVEqCnq42vK652avg"
    "rUcL1GisW2IPRV8vmR4To1EP7ZPsAQ1wxQEkJI2HyeSFdanFeNhajIITCvj4SNHW1sW/MxQC"
    "ghxQIJsSxfg4ZRpKKxuRtPY+zKSMjQtDNYBP1K4zQKX2QeRIAetSrEDZFv+3QoewCWooKQr+"
    "5mwbIbFpVypMbW0I1nfQxcmAizszgNSWfIUHd07BLygakXGLofRsRXbGI8xYJIUvdCitAOQy"
    "FX8GZHbFh9dWw1M1CgXnjgwIuysO8MUTZqfDaGjGpHl7IfMajZjpqzBxltWRiseSN0i09iOj"
    "tyoE2Qd20hQkmDM/BU/KMiG2pumgDYCmphCJ0WIYTXSBuwdRU3EB+tfVuJpr4fA3vCRoapFC"
    "QtfUd+jwvE3BycikvVWDwt8JQqK9UFx0HnJRsy0MbsWhiBQcj0LOD1VokpuRMLU3OkEhSm7U"
    "q3IPSFQEEoUAdWAP/EMUiJ1ioWTMhSxgLn5Mk+DvYuui9c0El14QZoRbHvRLw+uPzJiXFIrJ"
    "M99D8uptSFy6EnUaPf/2Qkfw2miGTNHDecCygltutpKzrqmH31M+D8f3pzfiswQ+/dAKUdxk"
    "ykxjAy0mKrzSlELq6Q1dkwU6rQdMTdZ/TAYPaB518LAwHtjkwS0RklZ4Yfy0NRBJFAgdAyTH"
    "CG654GBA7DQNDmTuQY98OORoxcjI2ejp7oKmUuA8aOiywF8u5gYFjDWjva4byojeTPh0kxrT"
    "Ezbx57D39+KLE2ahuRk2Ljg1om8aCiFxJlq/gQUpdUhb9BzvzCzmHwJiaMJpxah5CkTFqtBC"
    "s6RTrwCUBvpVbJ/gSn4r5cYVFObfs4/dbCDC8a0ga/cNLgTC6LhtqL0/ERl5UoqCCDNnhnKv"
    "GfxjIwW+iKmDpmJjJw9LX5n+kQwlN0o5YmxXtHlNF2dEFAZjAJdzJ0uRtacLWq1Vh3kdEmNF"
    "kMWVcaD6OXgm9JWm5m6IVGJs+W4VIsbNB6uoA8nbBpBTX1NPPjRj+caN2Lp/N2RqJQRPMabE"
    "+6HsHwuO5ZgRFj2a/+yrGOagrK1V8osR1yINxq7MDBaioWUBy+2laXcwMjyev6v8JlLwvLBh"
    "SyN/j/ARIfNQDQ+J2N/koNve0oaKhzpaiC6j6u5h6FtewdJZZQ+FM+m3GTFmt1TlAZ6syl1D"
    "TtZ1FBUKGBNgDQfjQbBahFZpz5s60GvE8Cj23oXOboInj1/g9KEr8A5MdxuCfgYs29GN2G/T"
    "4ScTMIwi/GslIw8hlMnIPdsLWExM/8kMOiMyqLEe16sxgv5qtAAnfwiFZlsdWIbBCRH7ZQED"
    "a85UMWLoDKSj9//gWAUS5gicB6wcs1CxHdE/MBvyEQv5dqzq8sD+zRLs2yLliLGgFfysQUVR"
    "EGgb5zQUTrMgilavynoLh9smDG5WfJISBH7XayW8M7LJ+jgxFEFAZPgobD7Qhdtaq0MZhRZu"
    "RO3jd7kRNDOIOwNYM8mrHoP/4HWz8AYXO0HVfiJeByxmA4IjHEkYMTkMvxzX8PJr24RYV0zn"
    "QUn+H/wfhqAj5H0Wp50u8rNW4sqZDhwtN78dL8LaMibMe2VgNz6Yf57DP14toPjeYchV4ZAH"
    "znt7Xq7LEIqZYn1hJdopCTO3J+PGnxYMl7rsVbkw7318lL2e09TMOZqK+lrXenM/EWHR2ks4"
    "8uUC1xxYvCaVL+7Eey4sRZmw9tzaklllxDCamuOisPOsxZn3fOynw2aOTnqW85LAKiBLNTJ7"
    "uNs2ioWBH06M2qv8YML+p43MYI5i5K+CROKUhPTEg5T1eWzT4LuXu1l4oaLstzUhvlKaMbeq"
    "wOrEQEaMDV8Gb1+ffpnAB54VpXEU2N0tCgKcHcv4kaz08gq3h1ba9BJzdyM/sl3aN8yxb2dG"
    "1Dwl9pi64oFtMrs57sdc6dnd+Q9qmgUFKOzvcwAAAABJRU5ErkJggg=="
)


"""Heroes3 Savegame Editor application 32x32 icon, 16-bit colour."""
Icon_32x32_16bit = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAMAAABEpIrGAAAALHRFWHRDcmVhdGlvbiBUaW1l"
    "AEwgMjEgbeRydHMgMjAyMCAxNTo0NDo0NyArMDIwMMsTHzoAAAAHdElNRQfkAxUNNyG10A1w"
    "AAAACXBIWXMAAAsSAAALEgHS3X78AAAABGdBTUEAALGPC/xhBQAAAPNQTFRF////bkIDTh0C"
    "Zi8DAAAA3MJspIEAPRcBAAEA9ue0////9uWmzbZa48p9/fvr49GHtYsSq4YFxq0Qj2EDnHUA"
    "7tuXe1MRuJ8K160A9+q6uJwmzrhCEgMCTykBhFgCrYkPKg8AlGsE4tEItpMhAwEBAgEBxao8"
    "ilIEkXAGe1gABgAAHgIAFQAAhWgJmX8ho4Uu9O3JoIQiyaxK/vngvqMstpIV7OGzAAAD1rtf"
    "xaE2pIILyrRPnXYKxaY82L1mUzgAvKZL3sd2JAcA+fLUn3sA2LNW2s+Zupsrr5Q61LRerZQs"
    "pYw4xJ84kXYjbE0BvJ0xz7NSwH+08QAAAAF0Uk5TAEDm2GYAAAIKSURBVHjajZNte5MwFIZJ"
    "ckhKO0whtGk7gQb6FqfVVTuZXbs536dO//+vMaGg0E/yIRcn576e85yTxHH+7wMAu3ahEbXy"
    "iwWfOxB6JQE7G7UBT4vfEIyykoBekvC2BuweN2OQ29G4BHZ6f3UCPOq9D1KGpYSJfhUnwMP3"
    "KchQhlYC0I+fs5M8uvwaCyklLSXcbw8neT/fDL6QcERTa7PgSkE7f59+8LgKA/rR2vSF+PR5"
    "1hoCoYfigKVa0UBm3QKnUatLmJBbpVRKl0JRmY27ihDcUpjcr/d+oSjSUWpdajJxETSB25u7"
    "jZsoopWQI6+LU/5exP80IOPrPT8QHIuEBkah7xsLrHFgwDOstUa4T2+oAdB1YpJRs8h1luSC"
    "IBQJZYB3guxOJ5mgvJ8goZBVWK2KAWrlKXX5m+2WTXPUN8Bw6F4N3jYB+arHz/ILhmMirYf4"
    "cuO/bloIfH6uGWNoqe0cXqD1S6SbQGfKYgcijJblcer5s4vnedYkEMGmkGDlnGC5WjFATQIU"
    "Q+bK0oSm5t4SlcPMmaNFVrcCCY4dpxNKFdlBkgmJ7a7IWH2lU3JsRqbS604H41IbiBAVIOIa"
    "CEwFHCVHIBjXgEuqbqX1SCaVOQxPqzlGdS1pLhyr8w6cH00G3t+dsoKojxEGw5JALnUriU75"
    "bHpn1RP2n/B+SWBMag2ol2Nkfv8A3KAxCtlEjPsAAAAASUVORK5CYII="
)


"""Background pattern image for export HTML."""
ExportBg = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAAMgAAADICAMAAACahl6sAAAALHRFWHRDcmVhdGlvbiBUaW1l"
    "AFIgMjQgdmVlYnIgMjAyMyAxMjozMTo0OSArMDIwMHDqD0cAAAAHdElNRQfnAhgKISSV89lu"
    "AAAACXBIWXMAAAsSAAALEgHS3X78AAAABGdBTUEAALGPC/xhBQAAAFRQTFRFECFSGCljGCla"
    "GCFaGCFKGClSEClaGCFSECFKGClrGDFrGDFjITFrECFaEClSECFCITFzGDFzISljISlrITFj"
    "ECljITl7ITlzITmEKUKEMUqUITF7cnZo8QAARHlJREFUeNpVfYmCIymyJCAQdEAq9FTRqdnd"
    "///PdTuImtdHHZlKKQA/zc2d1HuqKfVUeutltPRTSumlzfit9dZa7/G3nkYavaZeRx2tjxEv"
    "bqPVFN+Kn8ELR+8tfrCUPHPJJcW/2f+WVsqr9NTilaPi5RXvm+oYfL+WC/7PGS+NN4+nifeq"
    "Od60xe8Jv4/42N5SmS3Hu8Vf4lniseOtfuKrr/jz6PH38YqVxAeMMh7xbikepxWsIn5+9PjO"
    "iGePl8bf0tmwjvhzjf873jIesL96/HB8fCmzxGfV+MMz3qhxFXjIVB7xVPEDXE+sPPYv3jV2"
    "Aw8bL7qwZO4Ivlv6M34E/+ApOr8WjxQvmfMRrxtY7Bglnr3keL6WUrxR/D3FuuLlD3xyznEg"
    "uVQ9byvxkR0rwd8zDgRLGGc/Y3srzwVrxerfeOwebxy7+ypcWOOZ5B5HPDLeKxZT8MENJ4K/"
    "/uSWuFqIAvYyhOCZUsZ7xSux8vhifPWFc1svbHFqr47jiifii8or3q9jo/AlnHrKP9jEf3Au"
    "8SbxpJC72LxGmeKZYB1YwqCIYYWxa1xCfPrs+REPxcMJySuQ0ScWE2I3IAvYdHwTu7Y/uux/"
    "sP0QG8jpv/HTlIuR+JJcLKrc33iEFE/fsVPpQfV4xbGkEtKFn5vpH7zhatyMWAkEDivC8w8c"
    "cMjQqLGqgQ0dZ6vxlj+xVgoSpDlBBhIfLj304fiHOwxxbXybB7ZgQK648zo5fLfXjpfEOWkd"
    "WHmIb4heiCw2uDV+ob7ao1P6sBqso0LVQ3hiH7BtuUGyyrPHnkM2Y/En9ECCGns14tE7zyPx"
    "j9hBrCPOtvb2ibdJejbYjsKPypNP2uKs8Kyxp2nk2CMeC8+ycb/xjUIZ/sTXQxIfPG6+24xd"
    "vrJMBx79Ee/z5LHTOsUTPmIBA2YrQb3ys9FmYW+xJ9jgZw87A+GNhUOzuIyG04JMUeXwgBWW"
    "I+NzY2WDH0GpgTniQ5ZP41cgmdBJvGHFtodIQosy1jmeULefBgM12qNpobFd+ZD0xQ/lBiku"
    "UGt8yMjxe4Lw0IiO/uJ7lHLAwoX+h/GBHMAsN1qJ0KF4/HjgEK0zVlMgdtCuZzsb3gFPAOuJ"
    "z7B8xZMsCVq+qCphcjM0OlFBHi1X2B095KePFRr0rPGgs6T15FGVH5xOLOQD+awwUbTfsb+x"
    "GkgkRKpBl/WgWF/sAk4RVrF161+hExjUlc7DaHIB0Ip4Cxq++LlY1rNCXGOnXgVuhSfS6Euk"
    "4PIZcSJYP5SF+or3+YGDiF2Ix+54oNUXlQsKE+fVjpzpQ8JWw4wmahdEFyLwgJ43fzZNNcQo"
    "Z/hAfIiUNH6Uej/oC2n8pahnfcQbQWGpjjDj9GCF7xQP3CEJiRadIhcfyk3t3Av+i+dI7ZHb"
    "Jz4rRC9zPS2MJ/70LC88CR7joNnAB/VaH9yFJIV7dRrkE64k1vXC6iEUsh8Fkl/4JJ3GWSdy"
    "wovwUODO5GIpkDmMRsE7xUqKTAv9nOzmJxaQYQNyu0IaIBxYLO1/nB40KOv0f0KOfqhZBZqd"
    "bZpjHTJ68Yw1DiyeGzL04mpC81s6oQ3PSn8bi8kjfI52tPZyFligsLFhuWL/4WqwEjj3dsIv"
    "Qtdro6KEVMQzIg7osMTYhQWVioeoIasldrxd18yr8Aw7PRlOsUs9LtktPFadvVwXDNXkVsQi"
    "/0fnQfucqKAl9OgxaJAoPQNmttJBMAgIu3jgQGjRaUNpcytNVZd+QD1iY34qHHOFjcJKeAp8"
    "MMRWcoiyw5NuEVFIxGHTnq+GT206lqEv2dvEamK9cRwTGl4YmPyDlbaa4U0rZIvSDbsSmkG1"
    "T1A7etw4j/aEQc1Zlr21n/il4iMzPxAnAsmqjP0G4paOmA5+BAuHR5QLCe3jAnqTbKy9oHLl"
    "NbPiUfyHfYXCt72QBtHLE+dxxXpkI7IPi6rQbXTkpRHSxi8DASOeKkSkIqyKY5z6GKyEMUJN"
    "uUs+5c1xqPHLg+dS8Y4wIpDV0HsfBzQ7S0Hw38JaVqbwXNPPDCsXUhS/40igSpfP5Lqg5/i/"
    "8pg/fD2lDI9V4dWwlxARKDPkCt5i8GQYyccTMFijrHa4kU7hzg/KJ1Z8FlivplVAVhs9i44O"
    "UWGYyisVei+YVxmhcv0j2QqhmVeXX4OFxxriPfHm8U3GANfCwq+Qa3j9ClMGKelUHxw71BGh"
    "Mx8aVhPiRSkpOCSEIHSyTVrF5cPghIeNkIF6Rr3HVpyULwSxIbGPwTgQi3hEGsF1ywXwSCFS"
    "Vw49x5bM8IpYkM4YUoVtxWdlpg7xveuP3OiHoQiDZdkDGBGGEYjXmD3AfDaF0ilRbxkc0i1O"
    "BdSt2wSW/AlD88lTelOLDqEzI0I2wcNANIbQ6ScMX2oIgfL/imo/15JEXU1LkXlSRqZQXGoC"
    "2z8vOVHaD1gRRBa1IYOC7IddRJSB/xDp4BEq9FSWMyGOir2NaLEdIQ+92Yrk8id/PvnDj2H8"
    "nqQbibZLYjtwVPGeiNOekIhUduBIr30dONdtuhCUXnIy1MMkJzj514lADzo2KwUOCxwzwbbA"
    "wEXcqcSLwpAUsobhHAxSkoQN7hCmMjtUin8enxLnsQZkl9YVsa8MR6yCsk1NoGOAn4rQKRzf"
    "T1F44metfz7ccezGMXDgH58JpX/eQdkZbqy368+EtGW6VsUI3I4w9lU5WWOSyBCJKWGixYG+"
    "IwROWE/+tyEYiFBFPxHvFj55+nzGWWj74ACoJWPSAJWI9BnG+RhfiOVa8o+N0I9+G9jIVeVZ"
    "rnnQ1csoFu70gBbd66xLXhOOhtZpyDF0xYFQcdhR+sM4kweX5H9j4/8H5xdRc+Vj1WUxn/hB"
    "LkA6DhWp1zaysgUKzQq8CMxLd4A1Pzv/k/CH6Qonjz9QX/xVxhZQI5o1iJb8ckN0wBidcQ8+"
    "GAfENIM5q5xJCrdXEcKGn27P/O9Yi0kDopDYgconQ7ZWLFxYHIJ+CF+GIFPLFIXCASLvhncO"
    "YYblQDQZ9jQrHQoR/RPpETxGztdxTUdYyE0luhGfXJ+mRKwajdjLGZB+puo4pIpICwGnrBQw"
    "gGd94jUtP8KLfMNMPh8F+b4C+di6eNy8tB21V23zCAsUalapN0BDGG8DS2FoAnVVGgUfeNEU"
    "IfhACBUObx0LLjLenIADpYqxUht/rI+VKlGlmfRIecyqv2EdBYKVuOHUi0ZQIZbzyKMczzcl"
    "NEkSO0SY4RGQlSvv1FtKSvtL+CZ8UDkJ8chzEBFpfnVEobREcQyxps9Bp5KpJUVuvjW5amZ4"
    "NA8Kk/BuQ9a386OKg6eHEBQeEDUzUdlhfZ8pP3t+/Cy4sg+xtmf8j228dqAXcputLXIDXEpV"
    "kgjTHMfT5du8VOk3o6vYfSwnLMcVoVesZsXJ509p2y4N5fCtCVOjkhXhLNrTDsCFy4jwF7hU"
    "coaVCCHiKRCiRG5X6nMexXskLY03/X8XxdveDUu6HIUx4WOuR0c7iLco4/e6BQzOc13cYira"
    "dcjTAheJPboErIRuIJTGO1+2CvGDytmE6EkAiKApUk/0JMANGbkkYj6R3PRnOuaz0S03gTif"
    "eJLfELOLWUdkm91maWYJ0DklXSdcYpHZ6ndEHmIZW3+s8o33CH3HLqz1Lu3ASUdmAkOs+CQ0"
    "mXhX2tEhd6LSl8tAlTOeGxpShbdBiRGyDlnnghAlHOMjHj/UORzLv9ARavlDQRG3VkcwJPQW"
    "sHyGqDUethAzZo9nMUoVP3oQCc6Lsk2BLEcEjYjmJmzvcixMb6cMjZ95m3I5n648J3HPlKo+"
    "aA2InMBeZh9aqs9iMPBpx1X/pUnHQ19TaVbytwQtMdwHLEOLQzQFwgHMxUEWlOtaMEFICmMB"
    "+Mq6DBnOtm4RpFOibDKNUPBlTeOJtaHUHgYBWVFRRouoJTmeZHRyKg/aAR5wgBVRgtIIQme0"
    "sYqMsGdcc3VmakidAVDdIVZBArWwDOWoPJrrWAeDT6RM79yc8gASi4ci0N/SpVRIahSanosV"
    "BMEvohIAkX17n8b4PhcKYaU3C7OVI8KMr0eIEiHF+sRp/P6/tniwkF+KcLjrFR84y4pdvQYD"
    "kUV5jTiFfjcs8ft4fw/khUxRFsOro5zfiPFDnNpRj3MZq1OAO+iEChCvrPjjAt5F63CD1Vhm"
    "pX1G7jMsJBMQFpYAmA3AKAGKWPRPg7Vffy44rGVjSo0jzpO0eK5/cUOBUSTCz7Ka4VOpJIsG"
    "o+o0iM6FutCEtyNzlYqZFHWczaEUlSIrjck7NTKebfgU9qRvQx+/voQOO31mjBHnMmRlV/mj"
    "+A4xR7nlU3aZWTTiDiklvWffKgr7ysVIxCRqAHewlCmbF2uj2e5yEhU7KVDGqDaQsFj0Org+"
    "yzWBXMQyiIuAb2Q9Cr6VEhf6CKGK3yeUp8lWMWtwgs20tCseHqwl0WxOLUM7yx9FIYa4GHbz"
    "2irbQ9NXKDeTwzgVLonauj32MLQsRxXvzDCCKA6E6uzG/LRrcRzPMu68DWakEGYXblMJkBOC"
    "QFAUL/tDAYq3ml25XAgxA78W4R4cw1iFuCdxIhrhYqxZwkEDFWJ01Bl6nsOB5LPQiEwaOy7B"
    "v+E9inaoHmGC6xH+APEVwDtLNJcR5jsS7/QQYlEEbRM9Bm7FMsQWWWZvua9BC2nIkAUdiEHs"
    "xOrCn1YSKkbUC46WsqFKiJJXbEhYwDALhLSQ4ISmKNZVRqaYgCE8dmlGMPrLXbhyhcNc1o/4"
    "kaqshtBVuoswFFjGKYKRleXJJOLXuSMS7sNwmluH3P6cTUaAwF0TeIdwNB5a+tiaz57FwUlM"
    "SGo/OzSGZ2yEjq9lMTZ+ZH23T20R+TFQM5qu2ETAKQTvtW09AzPI4BCyjTdORHB3Ynrd0Bhd"
    "KhAt+mGpOqRO+TtwgIrMcehtq/KHrgImHiZcCss9CIGhTDgdliVY473LNxkAmCCcy3laZnmV"
    "EZAUWqiQi1lFAoCMFC+ij4hzRwzWlMu0/MnMJeiPKMxl0OMhpCdGAAc7ToUmTObjU87S61EZ"
    "/9AdV5fe+PqwyCtkRXAKjOxO1fhLeMvwHV/UIW3uGrwNAKExnbmVfkEan4hW4xU/rCHBcDGq"
    "yIKxEFnC1VHQmxFOR8Ms6VboSu0bNBgwEACGziFwCMgKqgoIY5l/cffg0wvytXINRovWHgcK"
    "2Nwq9YcI/m5z2F0QPppKpLS9FK3YR9pkVrbjD9lp8o5WWaEuRK0TJQDvjgglS9uQp48NInS4"
    "XCaMqFzVymSAT85KBGQmKXCREL/h2uYdcbJSqZoFIGlZuOPOfCqcM9O6omzZSI3fDOuqgvc6"
    "7WJjJTa92r9Essh8EEg/6GVr5mtmmLgk7wsZi/c7WHqiBQXoiKCXdS+H8EkFT4iEErnyjkjm"
    "2q4uNmcdN2Ah71SYgK4616R5Ri1elv6NWoiiUr4/kmpEwIWFyLDJlTYOMZgcffuhUMtwyZSm"
    "thZ+dp54bPIp4NuyUVjUQbuVnZliIo7LUxkqp5T4e57vA2l/Ktu3RIRSmdvEh52MjwolfrUv"
    "XlJNCWHomZnIxw6cyDcUd6HyW3+ol4nmF7oNfc8R8bJs2LVLPLmT4pUBG7SsBYadnZccdpNt"
    "isiZKxGHQIl0U/UH5XicCMNAQbCSfCARNJTNuXjqLD/ANWWm8ahXKkn1sTfB/txk/c+wphGi"
    "6wcxa8VpWfmLXdSgeIWCRljxnWubar0xnKRKG9QCCxbtcm0bZ9uwcDWiqzdsqqfPHWhArYhS"
    "2W+EJsVjFT0HqgMRsZkUo1IAQi3wSYrUDyaWJAeC9A4BVbjl+7LEW7j5ENjDBbFBm76EC8fJ"
    "55MuHalhZTkO1ZEQoW6Nh5ZknYX9UPeG8JcjAiKht+Ny+EfvuYD1Ujgi7qd7FbsE4toF3Kgk"
    "QdDt0+H+SUd6ZYZaaRB3JkhFwo7jxgiRIBuwOX3ZOYL5cMK0nF0nAs9/st688ZUuALQWVV4V"
    "PMOCMIqojDkJ7NMYvQWerPFWlKoac8lfcKCqy5QUl6F6xg4gnyW9W4G+pZ4dhLBEyeRMRXBu"
    "KWKKiBiORYPJBIRI/hjjOnEY/VaRLkIHszbuZ9qqwcUQvSgbCxmjbZGDs6JhxkrmXdfHw/4C"
    "JFAFpCLQ2gGKo328cXjEdqCo7HqlAG9qzCDgbWVr1Ik34jKcCZ08P4sFuLOyIokIBZGjo4K/"
    "dfbYyrPQKxlMFDwVBo2G4fgbivWyQ/4shgBWGwE/M3jBgFJyVtf7frb2syG7pEJGUmVWBgfm"
    "qxKSZMp5wC1OWBrCIUwQWKoABQrmc6iyiCUo1q+CtxRpypEW1ZiYYIpABziAWQuc7hAgIQSF"
    "HgTCteb5VghAWUpU95PqoSotfRayCZTeS36UEQfzxIq6nJncKKEqpKfxr0O4ybB1QmTPC0Iu"
    "WiDiLW7kyd3kcmn2TwKFY5uyIVuFBzny8dZBAIL89gko4wpvAm07cRKRvJz/ty4+h901/Fah"
    "6c1x1A1iFzZH1IjUHq/SH2DJudh+m8+TZiZ8FLwtrAnhVKlyLIDxDqsT1BLShyplrm0eQDcS"
    "Sv1gyY4HCSN8tXWIlEEWEDNnZissXghpbb+1yPPA/ojgxjIsrcxWlCSaCHg2kK/Nr2nNcb+0"
    "K7bo3RBPIC+IlC07y4n3nE21F1VHkVrVoeoYEjrBueZgVkVedaceg2HwOhiDv1lhoMZ0Z01p"
    "QiwJGKi0R/kUr4GZ/ilH1lWnbCq2IwBOzJ9Jb8xc6omDJIoMKH0REVlZOQg9ThXSnHAM8uJM"
    "r8Tz8+eNW1HxH41clR7HKbzfhzTlQHEv1jPjrJa8d7ni+Y5leyqsaKiUJ0/SHdVFBpIz3FUf"
    "CQ/8quXZ6qsiYKnTyQR/LF7+7aGVx0XhmvxM1sGma1QnoV+sBDuiHA/fIy+HOcfwWQzVNgBx"
    "MY8vgpIjIQxb8sbhzK4EHpJ19NWnowEm4IRjY7PgHaQEUBrk8mMO5nZcHlL5tItfqte4QPlN"
    "IIYw7yR23RVOfUk4oNw7CJKXGMU1MlpcBeo12XSKXxyq3MI7KfQJh3sd17HmAcMbRmkAywtd"
    "f9u/MZQJaaqmulRHrLVvJJVLBX0HxCZarbrTB2s9IYA4IRwGWD1FFeZmTHrs1w1nerQUOBex"
    "FJzg6K2GovrtyoAYEcUOCQqP+56m/ao8dLG61LJTgrZdiBB/+5EMGX+URxPltD8TYmGBD48B"
    "lEOl6YxqcYXF/b5V4zGRiyTEpadtG2EeEZeTxuASE61/2bDM7cV4dEy0mNq2xjpWxPkZPALD"
    "8VBMIN+CcLpRCWDk0OFTUFcpUm65W9Smt7WTFqpKOxgURHxMfJBfDNU73irvIIkGJJuaAnUG"
    "Irm6EO+Al1y6flPkCoFZHhHsxJzvElkKhGi9v4AmYgPDysd5Igu/xuaiMyUgCRRyDJDrokWi"
    "QXyxOipmOKyW0kIiYOI7EX1XrlOIxh+hj3zWnp2qdkGLUHrRC+tWMQNa0sfh/K8MMxAMyOby"
    "OwkAo5QY6RsQlgs7HW4Qib3pUWQbCP8dijYE/BmrockdpCeSnES9TrIqksCqGPIuAhKpwKet"
    "vPPm1Jz8t/3w0pJhakF3wMRiuKESu1Os6f1ljgB3st6QrPCPm5LWxWvedauuZyqE2q427m8I"
    "5mJZmRxs0I8JtGFRj61LO/wlGoeIOvTxADXGThiOeyKu2vlH/xvL9V3MxOckMeUE9OKh4El/"
    "GWW1+Qsfi9TnmK6IkAbxjvcOGQMlsBMggMdA3ezorss1I3bkm5GoiQcZaTwzjkCrE4Jf6g1b"
    "hJqs3xyrIBMuzyaAjGY+5G0Vx+sKiPjRRUzS1BzKDvEXQxRWH/lNbxbyhTW0+UUJGy4eb4Uf"
    "HjMbd8tTlcgay6ii5JnyCbMRtupk0D1YKOomD8WT/6DALOi7O5kQwPWLYBVBRZUGGm42nzI3"
    "Y5bpv06ESt5PEq7hyBipnEsoNvK692KDA7xje2eFkcfR5swmPLVG1QG0pTQQ2d4gDkI4H4St"
    "Tiovujl6En/2BROZynDc+F/1lUhK1+oIHh0zaCk3Ob+ZXUB6xTC+101ALFUlQrOTFAMeqqoi"
    "AX2XkKo8pxlqq/QbKN1YAdP5/yrvW02YshP1IJCT2A1AHADLkHczaaxbuOrxnu0gJIW6G78z"
    "RGAUSRn09NO5m/mQ9h1g3J5NpCSdV8v5G4ZDOQjK13CHsSzoYDFUmgFPTlICyXkVpxGUDmQf"
    "nSzaLLQFUd5JOqMIdb28BsOznnbZ5kGzVSVaQGwjVUzFvLCBMkUrKhUMxQ4M+8XpKeLSNyNW"
    "Zq+QnhKeo4gPTK0OLV74udDr0L03a0cyvSinLAYqE1BIFShYr1hxBS/NOFFn8B1Wa2fzgwci"
    "xqfggiRzSjoSnwD1TeIzPe2iPnlD4nk3V1+LUK3qmEXkEvcWxffeiFbDSvH44i1dOmMT0VG6"
    "FKzID7vLgkAOw3oU6sOgwdGQZTnE2EILCLFIIpKVVkAgcNvGu7BySWVGjYqERAcl2wgWF2X/"
    "hlxFwY9Yks6wOssykY+FI0QNboiHdWNmxTxnBjQuZhwE7nPVqRzuQ5mCwXGMp9sNSC5F6g2w"
    "AeWMDIEaLG8k421i96giQuqbKrrIrFRdLWLv1Dt1ImsHYWx10ZkWnwEXovbNa2EyScW2yc8y"
    "4JWFaBRa5/JO5VmXBKOINVUbKR/tQ8Iv5QlrYN/biXc9VdRIs4uQi3M5QEVpR51MFCGogAvB"
    "sBtltwnBitMdj25vbvRkGINAqwVt6PrCxBSSCRGzqVMHbzSbKyrFQljWqG5rqihp4yvTPHFX"
    "QivbiphNcD27OW6IBNjMnGX2nR07waZc6iEy/GVqltMQcbF1IoxDiXaVtoMN7FNdbwCHVcBv"
    "PUHxexdX8R2LEANQYHSQTU2MPdSlssoAEIz6wQWqBY+UCdmW5FoeqjTDthOxa79VmtUS1qVn"
    "N+Tl4oc+vorjRJ+oKkz7q0o0DQcCLMUaw7ELFPtQydkeg45THmyGOZhZeTwL6fKFYg3beDd1"
    "EpLXTnV5sO0ILSQddCJ8Th7OgaR4cbCr53fY9y64kA+RRaTnok6hDbQdjj6Bdk7vxRFZAXvp"
    "nGUNbVZxLYTGWcxb1YbTuxKQYCzBlGDbBIZM5Obw88hPCQk7c/9BoQOi38qLMVKcyqu60oKj"
    "7ZTuUL4dgFWoE3OwYj4C6Yyd7ShEs6kd54oNFwMmloX8LKvA59ib5okkCL2PMFXYOUgc1OZg"
    "st4VLSAVmNmGMPdPCzekvsiavPtdYBeJ29VKPEjxl4TDfuQif4qMdCrcRVAP9G0bHLL/hY06"
    "lFEQkyH4h7pyDBMNdhtSpSffwCUNBatEKOhte9uNKQRmmsMV4usMdzv7pnpo9AtiEoY31o2e"
    "FievsDTa/dg1pCGH8E3Zdx5Vy7sHzL58CMtBs6N2OO8aAkJRpmQstA12L8vOsUtvZidm3f6L"
    "ROy1UxJaUMA15qo0E1q0NjXtDRCmwdbKxLjoQCqTRaLNQ/D/moh73ncQJ8g5zOPUzvWDWHNR"
    "0UZs0H4oQ0bkUgU1XG58Fq0+FPTUM8FRCdkczVT7OOlVVD3t00lOE4LflWGZDkGSPBuKxyBl"
    "qEJNmJAO9mOlZivYQjeQLUQuFB4Z2QGjFbapfCMlATh8KCs9aehcX4s1hKBP6iiOao234jC2"
    "y0GIh5ipyD/rRkZoLxkaQpB/7+KWKPE/bBlU2Czuc3zeh/s9ErsuP+pcIDSHDl6YxiRWxRcS"
    "dI3yFaKiWsCiwvCtFnnWnW1K2ktTrkFC2USTTmi0uzyZ0ItzmtrS2aTi8xz2T92SKa+uTJVe"
    "j/Kn5IHGjp32rLYwh+hPSiYIN2DKo3GtsRiKZ5ENBNE8UkTyqS8b5RUSFTL3LQ5tk7S3k3dU"
    "RwSE3ZHxJm9zJb2e0FDaGhtUluV0nKq/S6EZcbuKJ7vMHOTW+MImf7JC6Yvx5GwvimN5sMfW"
    "gUJdQMngjN+NICeE+S7JsHT9Ppw+0wVOZVwVKrcmi0E4p+okcqim6QoqcGAQgEI+B203l6Pk"
    "IBYRKXSu3MRRhZw1k7KqkiyGp4kidz7YOQ15SmRLzdFRLeH2MnEo48Dq/xPLOX4R5hyMuPBe"
    "EW18I1klX5GM+E56B7ji/+foRA7HtXuXDaEICCMfdTDpgtb8xg+dLLl0v9h4Y3yAyXmMr4uI"
    "SGjtvSHJ+NY5gAnDKSb2PpMXNNourhEVFeuhz3c/Mkr5dR1QCWDO633M79EUHTdHXZnp9opg"
    "D6ALgAkxp4rw+GJQvm4aPELa73sQQmQxUpaiuwZTFnuxBXzzRJrouEM4lVEBAthDbQzqrVRl"
    "zDmTnW4mNMAadQZrjp3V+UAWyiSr08Tys8X9u2YEMQAPC0nTBjwZ3TX3GtEhDpoMvrCI9yWG"
    "aDcJc6MexNaLi1Dkz6sMJy+Z1KrKPFcNC+I0IzOMp3psGh+VKrzIfLOOQdyxgXol4iKYVAd0"
    "v0MxUZwDrn6+3291EBZhc+pQUxAp8DH+9p98NABzdpjlLvIDXK+yfgczGygJHVC4nqnACR4B"
    "Xoj4PgjZIVrlZEm5VCVsolqQ/APQoSmZeZcI+I5rfnE2b5IsC8KUmll1CkmbYJejKh8yb2DB"
    "mn02UzWYCIfXT+ikmmcEX7EntTiqhp0EZDSuUW4bfsCdIvQipwkJE5RwpyxoX0bnOWwgIRy0"
    "n4NQBgZwbptdZ/ClA2KeqE+vfPUpSCjvF4S3QCBGpm4OS4w9Lhv6bfYc7D3t3Vg2XWYd5gtq"
    "xdVVA/ByhjJj7iRC7UOFizv4ZZjnp6Op7eOBCDipXymhuV20NzPbpcgRGLYwpBPh1oSiF/Fe"
    "yKSPRye5tsOfhEc57o7i0t33SOFHowE7xLs68ovdPzMgYReM7Enmcc+YOyMiNXVp1QaAzXVk"
    "r7CKDr3ubiIRb65LPAUOqxjdiNcMEnXjUL4RLB2k/E6azghYJrHOxm4jLY/uf6UiffAMBiXz"
    "vXnuSB8mvEUYRbWg8ZVO0wp0R/Rh/96bK8BqDjFGlp8ZPdaePuAsJzdZ2pRRWtFbuJMXITBI"
    "677rjc1f72z4YR3xx7fmdbSiKJ8OHe0h/Sy7hMm+qc1/ZzwyRVjRP0pVEAHJWsFeZ+peUZXg"
    "P6TBDjEBWyE9QZmnIjAznzN7KFt+bKKuoCie6SSVHWsA0XABYzpu4BLFpW94+eOII5sRUs1w"
    "Lceab/ADxQiCX4DpZOVCmIxSQ5oTUdDQvCleiM6uGPMtpAuArFmJMBghc4jDxh3lGE988ae9"
    "UFxXU/zueOkO2NhYRRAeDej/YWp17X3k5h8kXxSWBeA6wY44CnN7bGc3y0KF6lTsu8yqQMmK"
    "P88CqzPD3XdiVur7nVVMa6qlMf4fnADSWadgq/YTSQjD+MQ48iO4BOXdTUA/rvJeX5Rf6Qzz"
    "zagXHvBFuMJaCc4Ueo6AnyU5klycvBtwrDJYQiUymk9BxnsTkgO67rrdbQzjzTmbpRobYpCp"
    "6BPhDb8Sn/jhdJX6xLSdUObMbc3O9SqXMb/QkJZ/v7EcxB4sfoBxMDPjlFRjHcAC48G+WOX7"
    "gMLUjLhbfAE1I/CPmbQrl0wqyF1sygC5Ld7kXEJeaZC6Euz8VX6SQWE4zeAoKuCywa2t+kGL"
    "wgi1Y6ylUmYxVVO4RljbhqSQeDib7Tau2bKaGcK3QiwYES/GWM3lki0I/ne4tbaKP7iHVZB3"
    "LkC8uNzGro4hS7trY9sdgeyywT95VhZ88LOfH0zY6WSZlo+hUCZfsEJUXgID+A9R7OYlGolC"
    "IbZhnSwtw+62vIuON9ldCINcA2eKdc3jMkFyMUub2/UWduLI1Kyyttst9kZgG5r/Odgepmaf"
    "9kNoJrHn4x9W6PVumXksdyrc3/EFI3/9LZYk9aHA1Ia+49jylsmyi6fnEp23V9JnazNJvhFz"
    "rsVNVKoRoahLK84pOAwKifRO1Srdmm9+okuG6DQdTtvJFylQwn9g6bopuZn9jMQbjqulSZCS"
    "Pb4I2xAFjaV5aGhoKb+XuY80OFXR8NjAdjOtqRmVFUpPnCo3Iwvkyclo7TiczQBzGdRUqY3m"
    "4xQE25Yc+8RnP8Ci+wkThlykHJ/5ucdgsSKyDk31qH8PuLC0NdR7PEGwAY+d2kEAbGpOQDbT"
    "QRhsB8m/aYyci2FJgDIbMIyacZ6X6ksyimzOCC82XaAmRip8JW+ipLOgolZYmC/yxv+wy8ID"
    "HwSS5J1cmUQJC8dEpYFEmdVGuHr5+2EzwodvtV0wNQshhSbZ2K3VrHwceABJn+jr27XVRj4R"
    "+5ighFnPpOKP2vsaozvFUMpVtNPphUC8/PlA5RlpEAedu2U4q519sAsVqM256FoW2wPzlaYD"
    "zHiyzpIMgVadbScsTuhnM+7r1idqAOptKIcdUw0HsAk0y+FhwLWZCIXaptHTo2cSH/mg1amY"
    "2DH48Pxpn2t+ZvWAjIpgsEGl2IfVnIAOrPI95hrEBICAmsvnR2s2q5HIeBSecymR6wyXGlvL"
    "s79ZRWK7AfNIiCu7YbJC+O1LYADHybaXPj4q/03Pg2BYnEiB00Cy50cYrpPMzFYiz5/I5W7D"
    "Y3iklnPP1bh714V++Nl26bprqoKGC93ztuR0ct1kDNtfWjMglOhsWBG67XZaEL+7SmCXsTNt"
    "79jFDYxeolf90NHd+lM8fq/cf2ONBWmv+RS0ZyDs6Xua6ALwfUdkO8+zXAk6aMLXSGQ/th3J"
    "dya2JX4y/5+7j5qV1tOcHFMi1KZC0LwAhfdMkk8XyzezaTcPvpwFBYC4TGLXu4WZInMSVR0x"
    "dt93KwUZwsfGB4tbhMom7fI3W6l28iFqRa9sU/OuxiMqyLFJQspemxsKADmdD4DNg3LBgThI"
    "qsLuhrnCCMSRn/3TPyk/QlYqLC4ZiZlBABISdIB0dt7he0LCK/ptWUkUo08m9djDAbomOQnR"
    "Rm2M7lzCxcZrtCREjMV+LYgSj3LQYRwc/6kdYeGGNCz4w4ly0HifrHVyJsRr12cw408lqZ4/"
    "YF4aN4Qpge3NCuDkowAJcrcufM5iCDD9E+xaXawcmzRxS5fbLTd1iYu+YMRzuydIFA9lrGLH"
    "aQnuA2Hln2dMaPoDkHrPQSuqXELgUn6Z1pPd0rSUU7EYWbbNv6lbrd6fPuEIsg0bWt/nJGe7"
    "FjmAsqdZiRW8p9EAeEOSO1VrnBa3xn5KR2KQRHG7u2uVm3Og3vmqAvaD0wM4dTLOTa1gHm7F"
    "NF2ttOt+9jZmcU+2/O6gal8rsa5MfHTh50KsAdMy0gSZK4uiaeYLdUGGNU7n7YpdiPmpFJ5F"
    "7TFE7tLkKtPUc9V8Iv0Aqa2uOGfSSjkwDngRa1woW4kUXTFV4N1cJAbMC0IzVpEVx08FGQCt"
    "J/UHPBKADquR5HcV6nDqBgOEWJv8lNiENgtzYHWbMro/FeT/RSN3uZ2EFgZvSQSKtjs344FV"
    "fAtBJJecH2Dq32NuYwpejhiMmTwHj9fg0EQUgAowZkgRoCD0ZmB8ABGivDigttxRBzZb+Cnt"
    "FQrP1RgoZ0upXyV1daL4cY3jmgNAvJFBsEadMXCs7YHjTmJjqnXkDpk0tPeT8+0O5OsgPyQW"
    "MtiPLe1Hro745tG+x1xM3UWIYlHZOlJYDknFhJxyiMzPWrbHe5EeVdVCLWLynvLQ3ZTNjnfQ"
    "TXL1Soq8bVcbfEKA89AYhAeKfW10TUisKrA0TzGGVoND9ybJzZk76Lo1/nuv+Yv1ogNBQ2KR"
    "ZWcPFuJwyiGDNIjF9N2HCo04MbxOj0zWXmJrFuer7u6frBmVmNemkWD8XlWZ0BJbmyiyHtlW"
    "DOxGXsKBBQj/FVe9qyZvxGOwwQCRPEQnUvrjy0Ty+y5Xp51W5aoiY7B2DJun37AKFysDhoOL"
    "+vJJHN90OLrAoSdlXy1eu9zUQ0ohThH6mvqP5+WZq9XEl6nM/9AQzA56dYfDjVxhnBhZXwRM"
    "GClSwq7v0Y5fwlAHCjsqAm1PkgWCYK7Fe5SdgrHcNPp2l4qLyyYNenjTTseqBz1MMiv6HrDI"
    "cRxi7CluTlIoRXEu3oGY/iE3TnPs5t+IMZfvBng5K25FlI14GwDXJGPy71AWxGVm7PYInFkg"
    "MRJo8VN5hxXTKhJHL81UqWEIA1Ac2aNVzb9KiCun1hTX5juJcpgHWx7k4bwQKl5MnlS12Qw6"
    "WOAv+iu/v2RDI/dBZrFCP8B95TydcCdm5aIBZao9qgzHSnsAq8dHVvG9xUpRE6N+cDC1VVGL"
    "hbpTaAO4f616dDUanV+ujnTRfzl6bip2wXkCZJlsswGtF7sIAzsApSy2oSElAU30/A+krK/f"
    "9wFtWd+KcEmtmdCFPDra2k8nt5W8WaeBmUR6PE5jM9PZUXYqHOdsDoxahTLm3CXRqM+xJ9sS"
    "1SeTiMwEshkLsvZWN2SJ9ezaRxZYTPAHdRzQHwCSAis9rowqTazjQCs02kULIAEUaUH50NkY"
    "QUJ/kycqDIMnTp/FQYxNTXUjedvmjr9NmMAzNp4MeIstXazf00y50UJgvKEm5Xp7mFYS6ZJP"
    "xx7awpQkEtNFpu5i22jjcA0QdYpnCKh2QgIjZoacmEslOuju2GyiporJIZBIOpHcgtLcgKQB"
    "HUplysZgO0ufSX9TMIZjgcnFfHQO2eUI/1ydjGqYRMjdKhii0dQgKrjpTb+aFWGtDo04KIXo"
    "kEItLSRRI7cutV9UguicpTayO127AHvl6iogAk5F72vaM1i43WOqw1k+k2MVyblnVbqxkSe1"
    "nbqoOSJMaBufSx8EYHKdoRm/m4wAUzXZasX8Y80rHh9U41b2QJYwAOea/fdULrrMSNMzafLE"
    "Jnpo+rSepVYLkoVMlbqqmdWFXIDKlnNyZjhEZZyDczORCHJSzc1V8q+k1Yo/h9nUk/0CHtQH"
    "ENv5OKmzKLoBA59Lw1YuzA0K/75ONfmBlCmyoZ6XhNLu8XVZaPRw43d13qIv1qJuvVCIotS8"
    "bfY4Z9UzOhtkXkK3OQw7yYlsnplDWzhjUW8B1ozpCtC6h7ihAcTusahqzbh6RRCw2u6l3r3+"
    "zslpaUZre1hx8yTkpkl23XW2LrgX6D8JUeKx2BXS77DzudJ40yWmp4iUqb3QscRUFN0/s6Tr"
    "UrcQhjTE566hPJG9gxEZR5zbMMPkuLAEdPOGchxoVpxoC4CJSSqRJwUQaqK+e2LAmdU1C66u"
    "S+zSyQsgOIOXYzHx1Iq3k3rTVHXRHnS8HMsghSA1DfROu9SifiFMBsokknMMwWCboMpMhcSH"
    "EZGuGg8W1oXCTcgYCkQF5fwTA0LdVdnJPj1d0SSSrjr6eZqeqgh3aLbJ0NQCxrnE4IvwXw8a"
    "KnXiZJuMiPIv3NtRxH6MTKu/iNeDNh6x1h4I2zYGl+c7/6JCTV9T3QInIAvxC2acTU/DZDd2"
    "3m0wbklh9bnsaRmisSczb3nvAWTv5MibugdGcygR2RHIwGDEq5nFlT51sAe9M/odciVp97A3"
    "TUXthhKy0vRrfQGa3Xm/c20PlbwWBkmS9s3YDL77e4oXr6KZKojCKjYHa8+i0OhY3XCige6a"
    "Js6TUWLjMt62ebSRuuCBsphgUcoPsp6f5pV8GNTQmGyEPg5zvQHIXhrv19zpzIJDCBv55oq5"
    "RPxWbNBnZs3jJN2ck//gWtQ9qWkUmtW1S3XqLY6I0AxrWidXEru9PB5Sw4K7bhZgpzutF7vz"
    "++5dy+J7tD16s5BYH6vA2B+FxKn8F7LPqj+gpv41VNdcZSZkUiX+CJI0LDcOrUtUIQpn9akM"
    "jdeGAms4cscQHH/TZC9oUVbjQ4S4GrCOrvzKaMt3OhB1UfsPhwyYIL0n95XlOVWGEYVMyvvw"
    "OGafG4cR2wsA082s1JZ0Dezl9TD/9e6UJg4mQPc5BxhqqrMjK9F9RNgo7pJ98B0HUyxwAREG"
    "E0v6uR1ivccQ7C7l+AxEVq7ruQS31cnlB2a/GwbRG/XpdTVjILKLmtZDC6MmftxilPYUYf1p"
    "0MSxW+ckROccmHla9egdtKB5UA5cx+imS2z+sIpDpdkksm5OEgTovVnZgVrjhmuuU7mg73Dg"
    "87LJLmsCEfk/xTdcFCc46FQgm9o8LQTlVdPIdLnA2ceeSqAuLZb+sNzZlhwpbZbnUXFEE3gD"
    "zN2f5JAk5f0o5CiiaGxOArVMTjwV8jOE0yjDaP34ZrIwu6pTHFAyyx38MHca1WzFomFQfXi2"
    "Rh3iCFrH6/S1TM1rEe4FXA7Jcib7ESWRzugSvSyJ5DkI0X91vo2izqxmUtVElD45fU2Ut3a3"
    "s2rFHGqBng0geQoN+y4s3AhwE+CjslFxW7Wntu8gitfpDE9C5q1Tp6cyjUvh1m475/BJyUTF"
    "Inn9RdFkaeaIVR06ya6iSaIjsRKRiTWtzV+SPCbAQuzIwZfBE1NjkFrwjQ0ICXEk1dwYrVE6"
    "iDdq2pc2CCkehMKsAoLsWrun3bjB3EOiXZVDliCUiIPy6H4HZrP1PryXsXlIN1ChVKtFuWkA"
    "RjiAGRGZB3GZ3bgi50S2mw3WkkdQVWc/TpYTNNNOAzZQx9FVBwoG9dS8numsapDVwPshm8pB"
    "8uPMP03bE6L1YhdPesgnkbId68hVZU61KPwCoydFntGJ+ZWF+kgeAvTPEFXRbUm8uOSrWYrT"
    "HAENfYpvZ7pBzhtCwwS4u3Io0nVOgW+q2tWu6Qyu7N7dQkCATk2BxaempJHWr650gVbtIbFQ"
    "k4ZGyZR3hOZHUZ2NAjh8Y0A7lZwQeKiO6FVkKwd1s2ooOod+7CYhHpgklNdlMH801t50Y4Lg"
    "uFoEFeF92Ict7hRHa8F4pO7ZWKLKPttmBjbWbknw56BLhrZFw5Jhg3f6bFCtm9g6+55rW7N6"
    "s2rx4D7jzm6dKhoZp3Ls8OzmegNvwynhrs1rqpmJKp6tQQhIA7gHCb5oWSFDAXYiYTKHEBv0"
    "s74jprqOXWJCzRvz0Vl0PrSXJwJpSH88/wB0PfdgxYpwparLI3Pc3uCINs3RwASqgzgla/cZ"
    "GWtRkj7aHvUEXCVl3bLHNvvK3mAXR1ycQCWBTSiIL09xlvl1tNUlE4/PkZ3NMg78ti8lfcqT"
    "sa1b8wyVv5A+95fvNWdo43y7uC0wXYBIA4R2eC4l1eY6az5PPksF24QzBnl8p4aqAbiOn7yU"
    "SvryQXY8oLiTnOYr1HBypcE+JAqT5bhwVQ6Spm+neqyu3tlkTMoVblXXd52usPKYiquespt1"
    "49OtaYRDNw+PAYRFSZ6BaNspg+gySVV/nCL8aoIh728zt63ptgumomwJcOTIqhaQq4pc8fBQ"
    "lGv76c5rsJQ4lU0OUDEO35u+kAPcBM9ZFBm+ezSaoLPdCJoEY41dHKkaE4nAY98YOU41kSVf"
    "ckRZOE2aOm3dU68vD89IFGzQNKjZnzvazW8/0RRIMdikbkyYo6sKK9johwUXbaxK/ygj2/b8"
    "Q97PyJIoegU4y08joYt72rjfmux42l26D9t5CMFuX9/EdmKOOxYmhvQwaWawmNjg+n5oTxl4"
    "QFK+DArNBlR3s1re0CLCqaBZI3LQvdKmg2S+iH37jIAYHu37OA8ifCzfFE2u6hqfEfKPekYc"
    "Pi996FpHV0+gA+LB4X32gh1G5MTdaE+VQYYYZJFBdV0ecH3ELeb1P1v8vRCVxKHLJzOouaQs"
    "JPsYJXP5kKJY1YHLhkAwDWdxef4uTHuCMQNgQSzdl7J5gJKJt/xq1X2MvuPs1JwEkGq6x2KD"
    "GQ4okZSoab+wLxi4pB5d8z0J3lUzXHcXatuD/vnPsT0fvYb7/0lGKrofqcizckh7dR3cnToG"
    "Jk7fkrIpQEQkWVfY03kFPQxcPfiDflboyqPkf7SxXfNb4Quvsurdq+JL3FQ3KWlf19l51ek4"
    "bhYqtXn1/eKkEQzC2ORQEep0N76zVDqKr9Nj8wZ3LA8P+3asjein3w1aGAHq+j1sCPpHGJeY"
    "Ldl5KA3grQbAjlm/+/IJFtfd1DhcVdIfYFDfKstnkTmKsXlA4oIQk9ueYJV/h9JLYnK+AEoT"
    "j0hoUc6G1rjTHU0aFIY/4Mo5ihw7dAccaT/TyQtVWD7YQxyyLzeDm2bz5L7ckuGRWEauOtNy"
    "yNn95Ua1nYUQGBKTg9GtYem2Bwt6e9pGN9kLdVON3GalXtNqYERXENAK1270WxnL2EPnaghW"
    "yw+NnSe3ZHMaC4N0NUqovtl2npRcoXcHqgdZdE3xa5seWQibGk7EQx8ebelhjpUFqqKJnqyC"
    "750zKb5VhcJGdlHkUS8+I2+099OPhpV7AQbO+ZdUjPxQuHTti70ExEj/RJ8ilkVCG1rfz+Jb"
    "W4j16A4ioQpIaHltEO874XRKdr3opCfjffafz+I7AFAJn1fb3etqFiXAYkjLWkEY78Q1h3jX"
    "82SemzSYLHQLk/0nLtEBS2s6NyyaUSFbn0hB89U/5ih6MG5bp6c7eFAa6z0Us9Q0gFYcOHP0"
    "jO4xd67bGqDGw1Z+psIcq3E2PTd85G2iTzn5CM1YWGRQmjTT3uUEFAamRgp83A6rE9nIrV1J"
    "3xPAk++jESFDF9ip8UgDKQziVdXL9zVXhMyrqhe+1oD85n5PXQHJ1qT6jUJsasrfwA0r4t2+"
    "vICS5NFKe6FheXnuC0wuqQRWUHz1Qill30Cwp0kyCGFNk12zSoJVY1m7Jbqwj8s1C0F5zWs6"
    "qnvBNJlL07DN0cB1X4xt610bIrjFkfuqXlXUsRj1vjDSMH04evvBVhKBbrwW67LFxcRP4COn"
    "TVLX5TxqnN2TzYjZ3thV/DqHx4ZV/uiwo8jt2+57rQguNY2Z54khQ2iajtAWEy03VSlTZDrM"
    "eeIJQ2FFBBtF87ta0pV0IBY2Mos/vMp3epAHoMB8YtD7IZraRiVsWGDvzyL7fiJWRaTKEp07"
    "F8ScU+dHKXtuD3SQULK6ANT91VzGHoKw6CqrbG4j0xSJyVnMeOJnkoX24E2teZM3WF51g8fN"
    "x+A/dTOJZegFu6tutO+0Zhi00wVkJUMlAXUQGEE2+UHMKdzNMfdiN+STEJp093XSOnvkTRIY"
    "kwp7lk8Cv2O4DZhXrpbdw1AU2O+Hx92Fvneuu2bKST6bWcpyjnxE9QSfuvFZrKMxuTAc0Xcp"
    "hSm+a0UqJLEnga3UnKuLMgIRDTfvDM3bHi4nimpQRzodBleWuTiqOGI5d3Hx+k+RguPT/kwF"
    "izReo3oWn6cvUbqVZzTB5npPfNCRNTsltHWS9mkiud5iISdg2D/l/dTFVzkWVbkS9/Tsu1bK"
    "zN0BL++EO9FjQx4dpOGcnW4ks4qB8J5JQNpJRXxpycfDQC2DQZ7akzx2KmmwN9Kdk3E2biDp"
    "EqlmNvbWqIpBY2qS7aI9yfJy2ImKSs33jEZaAmCh7oQwqV1MFxW2fQ0rx++TxYw6+7Ob0kFs"
    "jKmwicVFlx7te1cdioTgVg0+SuwhxmXBtbvg10wCGr4Pp3ZDvXcZgkMpmI/kud+zlnbPl9SE"
    "PnFk//a+Er0Tt8stGLzUoTJHBLNZ15A31X5ZY0jbN3ODTG3MZhU7cVF/mmlVal7GgQw5MQUR"
    "Ulx1so9tKjBqJh+ZhPAy1TSPQQ2iQN+3pba7iK0LHAkQE0DcfEElWLhso+omUQ5jYXiVB2rd"
    "yV3l5Cfn6xu7Vju45MVgge6Q2yOd7Bepobzqd3Dkd+Sf91Bj969oRgeM+lcTwerfBso2y049"
    "Ncan++r5UgwDelaEEw6GLQVE2fgwjp2qKSntQ4Qpwml/7cAWYjDYAzErpilqYu1J+tHwLTa+"
    "IY0QZhf/qGocgkcjjGYEew9xUkurAMdDgT5HZ1EArlk2xX+XJJRoqd52riGkDs+s67y5Y8gv"
    "CJ303ThGKlrXtRVqY2dUTx4Ts1yigxwQb+S+NQ9/24AnD13udk8PGexz9sG526j1v1ffuVeb"
    "l3DtPlq6t7xTid2KLbaw4mkNQ2+q39zcLUpqcem+1R1XgQrUr+ujQFE0IhqQ4XH0e14J1REB"
    "aJUVrvWmXPnShqH2+35+byd7+1t3Mu08rt95/3Yz0nflYb4UDvc5DkKpjVuFzJBjQNndVJy+"
    "wZH/sbe4Mq7eEB+5U3Y2Jwx2G/hvOtEKF3mByCMo/3H8o4gk7Jz23XbhAH5F1wPal0Cy540t"
    "2rf44pLKlu4ZaLOJNciiosAAmxi5rJR0d1U4kp+R1GjiO8z7k71wsZzPyu2zQP/PqoOrOqhg"
    "8eyeah/J1bmUbPiiBV8nSJxQQiZXwJSch4C72XR7AWNsMvR4qbNxpzwUiwkF95iCfc/H9AcB"
    "i2caw5pm85AgBxBjD71qYJ2zi6y9VSRTY+TwoDYpnSvKRPF7ckuBpt6EDZPKnLWrTksuRV1U"
    "WoO6t60qLsxXJpXuAesisDP04MjrPYtYE9s8pIBVBZjf16O/0A/TNL7Y91kptWwaR6ib3igZ"
    "Gltn1rFIC6a+byC66TIoscglV45mI3BNGu7hy3nt9BlRX9zI6dGb7XYkvhjFy0rGohgsMlHX"
    "bcipvUSwH5wYtM+RJqqjf6xd61J8kYpbN0rzg6FrYXg1YrTRbaQkOA6H9TAZC6PqIyas6C+8"
    "h8krs5XlY+NNNSLVTJdQfuayKYdx9YgJfee9pi5nNLcyp+HV7BBXaE3vP4Rc1HD6b8U6uDDG"
    "5iASY75uMa+F9zVQc9JpPhKB5Qc5/T+NeTr10f6rH6nNu+GIK8lqZtMTq/GDc+ncO9LFsyDX"
    "6NxZRBOfKykRwS1UoGgqTHT0TpJne4Id1D1BXhwfl9XZU69+WmyOipeNg6E0GbXr7TqH0VfO"
    "rU3aZc3hlGXq5o7yoTxCx7jyJnyU25vsablts74ors1EqCJyAwIkzTTlxUXwhj+6GZSP6pRB"
    "AJBn03WP/GMg1U/SLjE0TZ1t6cHZVqht8v3HY7NSBLCOS0GHOw507bpLPvvqL362rpzfTBc6"
    "hGRxY5TQ0zClK1GmI40QnygpuYZqaubdY8+2/exMddzDNbkiXW2R1B5SNMYqfMAPhu4Wzo4J"
    "9/QYnNfl2QJEg4cwMRh8QSz17z29/Li/seOGIMv/gj2sdSSkgud1GsbmDupKRBR9Bi/c5a1V"
    "uyNHs/g1G2d4pjHvfwElp9q/YD4XoSVqOKzYE6BKu4cuK9CYfDjACwWcx78XXhYl8LPuaL8p"
    "5yZXHH2rDsBElFBgBPfq0gJi4G7vCz528hmmO+reJfMk/IkwtMNj+CBKUddEqVjgjybsYmt+"
    "PGWi5r+3Z2pGr+EFFdn1DTSnGbfb26fbiggVGPgtQlJYJqwc9/7SjEZY4pF8Gx/x19SfGPli"
    "5ovi4lR3E20/q0MxZTea102qISRJMEZRmxNN5EOV5Qhk2NqEBixRmeDkWQoYZtZ4OtQu1Yva"
    "meqOCYVruV/MsFDCNK0S3uNMulcnBOmhccvxzefjyWugAEIVphd7zs8+cNqZszZ3/VPZ1Q08"
    "Bi9l8TwZNBjwXitOyN1hHKNRTJ46BebWUdt9BGSwaO5131Ei8bMuplIRrR8lBM1WLv0Vlh9T"
    "csn81S4K6csY42bmZ5UHFRVV0wfJJkhSmmGkl3X8FAcyMk8c1zBxECy/l3TJH9m2eXw8aKKc"
    "l+g1JtJsu4uf5CigYpbyHulMhi2g9rTv+kmoRz99bwgnhXFpaN6lkc9q9BE5x7V4jr3FWsrw"
    "5911pK68b+TQ8Meei8L6Ssen8vMzC8KT4P7HRyuPxFi5uo3L/RF85kTH2jiLxLdQWlVN+AIo"
    "+Xo9m0m/ZGQn8WU7JVmEfN1Qg6+zqFo4qwRDbXg0ud1omzvTSWzBSammknlTWcEcet1Cx8FV"
    "19RsRK2jM9+iF21/OyY5pWVk3xym8jtipWTPRnIcKE6kTpbxShzbOKjqzO+S2Do6OZVSyUrX"
    "5U2PxEg4ESO4AcNhKo0+BBlHHmIKFtE5II0YpfNBO8mlCzF9nqXpbkVBNgo7PgbkqquU4r2S"
    "UhbqE4ox+6MyrQXujjiPF1l0VR2QwDffePEs5U5ZcCQPc7of0IqwAmTnqFNZccRuIoQmGU4s"
    "5Z/dFovXLQ2J17QJej+Vr0QFEnCbWvsbsIApU5Iu3uPIcVuCiAx/0IJV1GQSAv2Ue0sE6BgC"
    "8SaV7uCMfUzJszYpATSxKEUXYVXVwtrFWMmluDegY/YbMiFdOU02BJO/MmOzAfF/2g5kxdMS"
    "a7zeYxsc0Pnmkq60hhNJ7VhSk/X1h9Oo8oY5BHa8q129HAyQ0/Dwj6bBRAl3nsM5lOSZLYjI"
    "xFh5Me7XjdbNd5UuA2+6hePKn59P/vdZnhHI/uR/Gb+DO81PYY4yxDRVnpAcBpDtksSEReCd"
    "uefdFSfalQc1ChFvp8alolq3CS64BkokHdykVXalmweC+55LOT8aAY8JGxlW9kl3kUakL0/R"
    "fZtncXn6C64+j5WM9kM6axJ8j4squlry8Ee25pCsQ9fNSvOjejAUC+A/rBFCg4tnCSoK4R1Q"
    "iLtJqXuayRAf8vybnnGkvchpME+PZNisGafptJP/Mue4oFG58RoOwki8zZtS9nlez5L4gzKY"
    "6t6p5mgM94U9hh24iHG3WVHkQj+R+u6ALmJNJjLm4OUT6XO8N0lYHlcHCi9vJcRHvB66z04d"
    "vfQMrq3hlCm4GaahtD95V+04bibznuSw6+UT5/HADVPWDTdPMYcg/5+76W50icMuxLvu7biX"
    "N2Q/yh6QmEQMpI4o5KLBI6T90NxQ7QfGJb0QQ73IUnsp8lMpvHvUkS+UiBetrvkuRTMjMnt+"
    "I3T9lE/CTGGGrImKmZQYK6nwZWy43DAJX04kSKh+Wf7xRdYO8R669FONxK9+t8uzDQbJxIsm"
    "Axgqfnsw0X7y6pvyqhqA5M7OxjqjQI2TBZWR/vT8QdWQ94DwTNYDtyDkWMUTU6MKbkKIZ8XI"
    "5LOLsc9iJsNIdK88KM2NiQQkJusWaQ7SLuTGMoOtL1lKaHf8cfyDJQAqGBpjkdNPbMZP16WP"
    "D4KPP7GNCcM6Ohs2hvrvisaKFgZV+YdXluBy9276Q1ctR6Kcn9BLZm6sWCJYHW4bo5orhBsa"
    "blz0IqZeKihz2AxbgaDV2Fnel1188Y6uzWY8OB6ko1Khw66qT4lPEWHHo7I7jjN2m2ZTVl3m"
    "zoAKK/cVhH/byOOLSzOhKF4rc/qNrst6NHkHzZql8Uyqyva71al6coMA4kzXiHzad388ion3"
    "rL+U5hsJ1FqVNKgqZO+l+2i4s3jwWp+DSQ4/WGNGdROScpU9CVI3e7NOUcr/tIf5dcUG5odV"
    "qExUoGkMN+RHNyNAUZ8KOHMxVESaE6ee0C0P9+9Cnh64QNq1LlgzoibAZMmvxY49UkjHD8ON"
    "UPD0iud+aHgMFvdSNDdk94rqqb46jCGcCp2TRu3gYTRdKP9kP+mL0ZrggmpWOphlCENOz2tc"
    "ed8a7MoJzOLuFFGaFrL1bi9EQS9cBcEppiKKJA8Mi6VWQXdN29OTEIfEvENXgauCwD0iRwew"
    "mMypAGQzyv9BpziIC3nPRMoaVSECf2KvbeKtNAzscspl09VVH2i4xjRr9E5NmrDMjWUot8i+"
    "N56gsTvt/wMHTAvqq/dmVgAAAABJRU5ErkJggg=="
)


"""Icon for the Hero page in a savefile tab."""
PageHero = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAEoElEQVRYw+2Wf0yUdRzHX9/n"
    "AS9OOQVKCRINkRSl/BUyDdc0nYkup5FKy8RmrbA2Y8MllZqDVdrSOZouNy1n5fy1kEQq2RLj"
    "ODUz5or0Qn7IAeePe7iD447He7794c6tiXC69Ze8//luz/b9vN+fX+/vA/3ox4MOca8XFCEk"
    "wLJ5ChvXvondfpmGpjo0zc3az5vF/yYgSJw1U7KruIgGey1XtACDTT7aPT6utjRRa6+n3tnJ"
    "vh8CIccNC5X8vRWRbPpwI53+LtqcDq5oARouVdHYeB2AhJHxpKVPIw246Tsu958ILbmwUMkX"
    "Zc1Hj5nNwEGjsB6ayrlzNQCYw0ELgLfGjaa52V9+FXNAkJslZfGBvkWIUAT4XSWoEakA2Ery"
    "2LrjEEkJFrQONxGqiiliIGU2H+aAzhvLU/He1BkxPJFde45x8KQU910BRQiZNVPiqrcSMy4d"
    "YXRy+sx5khIsONs8DB1moczmo+1aF5/mT2bZu5XIgIYwOrGV5PHEqEhexCP7EtGrgAuly6Wu"
    "HZW6zynrrPly35Z0uXP947IwN16mDEbu25IufR0t0jB06Ty/Ti6ZhUQgF0xVpCKELMixyF45"
    "+hLxT7MT1WXHVbuVb7/aDsCIESl8tqOF0uP5ZOdV4bZvR1HDeTuviOGPqDTZPmDD+qW4r1WT"
    "kZHBb8ey5X1X4Pudc6X1wGLpqS2SuVnIvYVPymhVkYbeLnXtqIxWFVmQY5E/f5MpDX+9NPR2"
    "WXvidWk7sEQaers0DF0aRpcszI2X9zyEihDytedhxvRULA/H0eHRKCmrZldxEdmvvo+7W5C9"
    "MJlV66qpsxawYlUxAF41nEhT4HacgNfg1EUJ8k7OPgVkpgmaPWGseXkMisnM72fPUFGjkrdy"
    "Ek+NSQZg2xd7sV2QRMWoxEYHyH5pLq2tDQDU1vzFtsMCQ8oe+dTeBEjYaHewYfRjEDtUJXZo"
    "FNGDhxBl7uDr0iZOV5/nYMkfnKkLY/JYg4gwWDg/ncWrfyJKvciPZeXkvZOP8FqputDzJoTk"
    "hM9MGIj9bwduzcP8ebNRTGZmTA+npLwSULjsgtYbKnOmxWJ+SKHiu4V4brgYPTaZ+IxPgnnK"
    "niqghCLAEhnFypxFONs8FG05QpWtmqLiKiJUFXe3QHfrxEbf6rnXZwDQ2NhGY30zqxcIICDu"
    "1u6QKuD2uJjw7HMkVZ5lytMKgyKH4WuvpukqTEoEywCFwydVoBkA6y82GpwClyZvDd9dsg/J"
    "CTPTBCnjUjB1+GhtbsDtsRATF8/MOekADIocwhSPhslk40jlAAK6gzGjHiV1komKU5cpmGah"
    "cLdb3JMVB5/ezDTB0qw0XsjZRkX5R3R0CRyam4/3/EpwqTPTBCtfSWdiahx+fwvW2jD8/lZ8"
    "XYKkBAuFu90oQkhD9jyEorfdT5k4nuTE8bQ5HWzdXMzosYms2VwHQDBg8LVMGZeC4fdSWlHD"
    "pZYBTB6p82WZBKkGZ4CQBWx6Kw6Hs5nuTkHr9Vs+YA7oeNVwzv3Zfce94KMVNwQcGng7FI7a"
    "jJC8Rtzlq0SqIAK3DQHxn7NH4UawMfL+f/n60Y8HD/8CJkMWqBywNrcAAAAASUVORK5CYII="
)


"""Toolbar icon for clipboard copy buttons."""
ToolbarCopy = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAABQAAAAUCAYAAACNiR0NAAAABHNCSVQICAgIfAhkiAAAAhpJ"
    "REFUOI2tlE9rE0EYh5+ZycYQsMFLv4Chu/F7CIonT4JHwVsvwRL/3AJCiZQqVpGC4NmrFz+A"
    "mhYNaqFtQvBaDyrYtLahu9l5PWwz7ZpukoMDC/POO/Ps7/fuvgP/eSiApceNG4gE059SnYXq"
    "3ddnpXIAUTjw79UeTM178nTZz8rlALROgl7v90RYqXQBPSafG3dYRFQYhmYYWwued4gFmRrY"
    "7X5zsNbXTzcP9vfLWutjgEIpEBEeLS2maqS1VkrxNlNhFEfmz97exdrCfdXr7apxTgBmZkos"
    "LTeujADn5soA9PtHNJvvJnGGbtw8w7IlDMPUxna7nQkLgko2sFweKuzTXP/g1oNg9DcVEfec"
    "CbQWut0uwIjCTqdzBgxAqFQuZQEt1lo3Pz2UUik1w3cFQYV8Pj8KjOMYsM7y0VGfj601t9H3"
    "/X+AiULP84iiyAlwwESdizgtUETY3m4DJzZFoFgsUiqVmJ2dJY5jROQEOBgMUla1TgN9fy5V"
    "N60NxrgmSiu0NgEWCgWX8DwvBUwuJnE5pTQigjEGEaG59h5rLWM7YLHxcHCnWjO93i4ioLV2"
    "qrTWPF9d4erla/z89YPPX1qIlttjLwfguDaJyqFqrTVKKYwx7HzfYXNrg3PF4kp1vvpyIjCK"
    "IgqFgvu6xhhnM44tm1sboNSt6nz1lath1vDyuTcvVp9dh3S/AkhST9s/iM7X6/XD4fpfCgkm"
    "xF/LsWcAAAAASUVORK5CYII="
)


"""Toolbar icon for open-file buttons."""
ToolbarFileOpen = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABHNCSVQICAgIfAhkiAAAAa1J"
    "REFUOI2VkDFMU1EUhr+XPtE+2kDRpQndcMCNSU2MDgzuhFSCAzEMEqsyOArRwcGFxY3ERiQh"
    "EIgzEBgIm4OJCQQ2gyYmSmOpBN99XN69x6HlleJrwC93uCfnnv//73H6830vstnsS06htQbg"
    "UB+OTBWnJ0/3j3GejBbkwdBwbDMIFMV3b9mv7A8uLHyYjRV49HhE8v332NreAsC9kCCdTjUz"
    "jChXfr0qPBwdd40xJJMe7W2ZqmLCMpC/f6bA3PzMGDDuaq1Ryqfyew+ATEcbAL7/p+mw57VG"
    "dzcMw4YEYKPmdu1bJ+nuvtZQu0BsgrjHcZw7QSqVJpfLxQso5bOyecBGyQNgYnWmid+nhqq3"
    "UBQXIJn02Ch5vHl2l0CbM2MDXGpJ8HRiub4DgB97AV92m2//GGMsuSvVtFECAAFEBMdxEBFE"
    "wIpgrBAa4chYdGgJjXA5fbEuoJRPV0ZRPtB8LysQB4tQO4iAIIjUhIHgyNYFfpZ26bl5i887"
    "ldpQ1V0ELILggEiU6vrVDpYXl2gPvk05t4der/ktnXfOtbkTePrrx/X3z2/879w//AU2KMqJ"
    "njkj5wAAAABJRU5ErkJggg=="
)


"""Toolbar icon for save-file buttons."""
ToolbarFileSave = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABHNCSVQICAgIfAhkiAAAAopJ"
    "REFUOI2NkkFIFGEUx38zjW5qk7uBo2tbrplp2KHCcC0o6lAdkig8hFBYBBWW5alDRXUohArE"
    "QIoOiV2CLQqKuhZ0UIoQou1QZomZISm6M6y7M/O9DsLaZkH/y/vg8X68//992skrZ89/scv1"
    "iUl3WkOtAG03UGdoKm0uTv0sXpzpvd959Rz/kLb9dPeh6JraaEWpNbx+lXUmGirY6Hg+ie9J"
    "3gy8VB9Hxqk2EytvX+379ldA28XOHndm4oQnOuN+kFSghEVGHi27NlNVvoyuh6/wR98SzZ9c"
    "MCzCZdo62mVkbEhERIa+JmQgMSTtt59J7Ng1ib8eke7n76T5wh15P/xNfpdt29J+pk2MgOGz"
    "vDTK1NQUwSWl6No0BZIB4O2ncawlBj8mk3wY+kzZ0gLQAIT8/HwADM8FUYKvFE3n7i5Yc2B4"
    "CtfzuR7v53q8H4CnVw6DzPV1ACUK5Su6jzcC0HpgH42xemZnHcKmzpbYJg41NwFwcMdqlK8Q"
    "kXmAiBAKBSkviXLjaD299x9RE12BjoaORmxtBX0PntCyrZK9WzZRWFSYCwCwbRvLsqgIV2ch"
    "jeuqqI5G6Lr3mJZtlezfGsOyLDKZTNaiAaCUwnEcACzLwrIsek4V0HErTtr1ObJrLa17dqLr"
    "OjMzMziOg2ma84BAIEAkEskJb0NdAy9uNgDgOA6ZTIZUKrUgZANgcHAQ13VzGp7nYRhG9v2n"
    "QqFQbgae55FIJBgbGyOZTCIiiAiBQIDR0VFM08xW27axbTv3CkopiooKycvLy34SANd1ETUH"
    "EzV3PuX7uRaCwSDFxcWEw2EA0uk0QDbtmtoalFLU1NailKIsHM5a1k6dbrukaVxcYPI/JMLl"
    "XzXUOEhXykcYAAAAAElFTkSuQmCC"
)


"""Toolbar icon for save-file-as buttons."""
ToolbarFileSaveAs = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABHNCSVQICAgIfAhkiAAAAlxJ"
    "REFUOI2dkk9IVFEYxX/39VLHnHrj4DMtc0ytpBYRGmOClos0SKJwEUJiEVRM/lu1sDAXipCB"
    "uJCkRWIbwaKgqF0ktNAUN5Et1CzJMiTFcQZ1xve+FmOmWIGd1b338P043+Gq6403bn4KJGvT"
    "M+E5hZ0Cqhg4qCt7yRmz8GNHTKizu7mpjr9InahuK/fsO+BJTTTHD+81azwux5HgssXwt3kG"
    "+3vtkYkpMp3Dezqauib/CPDVN7eH/dPXlkVjyjJYiE5gi76VsqJjpCfH0/r4DdaXITxRMxuG"
    "RWjAV1slE1/HRERk7POw9A+PSVXHC/FeuSM9AxPS9vKdlN66L+/HJ2WtAoGAVNX4RI/WLXYl"
    "epidncWIS0RTczgkBMDQ6BRmnM73mXk+jH1k53YHKAAhKioKAH05DGILlm1TUvdgQ8z+8VnC"
    "yxYtPX209PQB8LzxIkjE1wBssbEtm7aruQBUnD9LrjebxcUgSU6NPG8O5aUlAFwozMC2bETk"
    "N0BEcLkMkhM83L2cTWf3E/Z7UtBQaCi8Wal0PXpGWUEaZ/JyiN0Wux4AEAgEME2T1KTMVUju"
    "oXQyPbtpffiUsoI0zuV7MU2TUCi0uqIOYNs2wWAQANM0MU2T9koHtfd6WApbXCrKouL0STRN"
    "w+/3EwwGcTqdkX9QVeOTnOyjCCsFb0IDg28jCXK9eSCyQlCRhpWAqJVrBJ+ZkcHo6AiCQq0F"
    "uN1uXve+WjUE2ZDmeH4hAPFuN4ha9XUAwzCIiXH8M67hMgBwGa517/qvQ3HRqU02EJGqrPbd"
    "Vor6/xkWoeEnOFf5pyJyIZEAAAAASUVORK5CYII="
)


"""Toolbar icon for folder buttons."""
ToolbarFolder = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABHNCSVQICAgIfAhkiAAAAflJ"
    "REFUOI1tkj1rVFEQhp+7uwkBC0FIIQbxE+wUK2sDYqWV/gELsbSwFGsJWIlpra0EQRQUC7FS"
    "CxELNZpCBcElH7vZvfeeO/OOxd277k0cOHAY5n3OOzMnAxj/uvdUmS6grCszQgGRITkhR9Nj"
    "6yGtHjx9d4VJZBGPuuPfP2x+/wk6c+fINx9TjjbY3Bhw+NQy3bn5phZLO/z58mp96ezKsSbX"
    "ybKrXqO6yN6yb/EaB47coqctvn54TbnTJ8KRl2RkEH6UmeiwK6x4BnRxr7BUsvnzI6oKVI3x"
    "aoRCrfoeALEb47hVhAy51UIvkZeE/gNQ+DQhGaECtwopIe9gaUBYDZitnbbQUCVDlpDv4G6E"
    "J8ITXg6wcoAVA7rdHt/eXH/QAshtKvaqJOXf69zEthUDrNzGym16PRHoRhsgn4qtKhhvvasB"
    "lpCliXiIpSHzS3dac+g1DhqxpRxLOXIjvESeYWlIeEJe1TOPaAPcrSW2Mq9deSK8g6cgVG+l"
    "BvxzULdg1hJbypGEfLJKy5EVhIyb77f3AupP80/sVQEREEbIpi8D9PvjVgudaQszYq8KZtc7"
    "G/eXbhPSWnsGqXpZjNaX3QWRNVzEAs4CZSxARP1yaI2M1QaQNZfPL64cylNxsarKS5LO9/uj"
    "T+NhPDx5fPH5mctP1vdYmcRfFHLMM/uOqXsAAAAASUVORK5CYII="
)


"""Toolbar icon for clipboard paste buttons."""
ToolbarPaste = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAABQAAAAUCAYAAACNiR0NAAAABHNCSVQICAgIfAhkiAAAAmpJ"
    "REFUOI2tlMtPE1EUxr8zU1CUR0tpKxKMsQkoO4MuMMYgAUl34GOH/gMGE3YQFlAjS4FoQqI7"
    "Y3duNDFpNCStLhoXsGBjfKQJwQbKQ+hMp2DLzD0u2mk6mUHEeJKTyX2c7/zul8kB/nPQYReG"
    "hoZiALortuKRSOT6XwtOTEBajqO64B+4KB87+QYgX2dnO4gIzIzFxa8AeNPI525Wb7xeONuN"
    "QjgM4Sg43CvdBfFToaP2l7dLCrR1UWtrwEaRSm0i/S3Bx7cSgmRowqDhuZh4CQCuSrJMgmcG"
    "rwUb1vf9eJdqh9frQzDYVu5NRCAC8nkgnW6nmpYLcm99rOFtIjkNwCq4HEd1vQx3VZWEhe0O"
    "jI+PYXr6MZaWPtsI6+pqMTo6Bk+jB1+eXwGYmgCGRbDsARO0POD1NmJqauog78HMcLs9ILbu"
    "2wQBsxcQjUYPEAP6+284njkKmhEKhWxUxRQQQjjWOAuyM6EQAswMgNHX1/fvhCaVYRhlSlVV"
    "4PefstVIdjgu0xiGAV3fL6VeyuI6mUxWPuZwwuLzhMW31dUUNC2H3d1dMB/FQ5tg8buy8gMD"
    "A7fKdxQlcxRBwyLGzBAlKkXJQFVVaFrWVmfzsJLQTEVVsLa2Ci79KqqqYn09jdkns38mNA0+"
    "71rC5MNJx0bv5+cBALdbPuHemcLBgi8+IP+gB9vanu67dFrBZf5oa+QUmZwOCN4w15bxdb8H"
    "gyRJczDYJyrOJMk+N4Uo9mHQDp8IPKppu/psZubVnuPEHhm5U5PNyueIqImZfUTkFwJNRCwD"
    "2GHGFhFtCsE/XS7je3NzRzYcDgsA+A1A10wi+21ylAAAAABJRU5ErkJggg=="
)


"""Toolbar icon for undo buttons."""
ToolbarRedo = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABHNCSVQICAgIfAhkiAAAAg5J"
    "REFUOI2dkr1PU2EUxn/n9vb21howgVohfvCxoCwQq/EDIUBMCAFkMFXjoIvV0UQ3E/8F4yS2"
    "CxujBINhIBIJOoEhaFycxEFMjdjecktve+/rYCnQoiGe8XnP88s5z3tgHzWSMNRo0n96rzdt"
    "PwAAT8ni0Fig/78Bt2/E0TQ1O/I80LtTl8rGgacEdDPQI6jrQKcIHVsAgPGJBCjpm7qbn6sC"
    "jCb1s66SR30YdQ39lxqM+pYWRPD7TCznJ0tfp7nYHNsFKQOuJI1BT/H4Vnf3GV+kVcs431GZ"
    "j7um+5DLAJQh7SmnSwCGE/52UTy52dN72YycwFqfw1IGQc8BoLZos1AobgcnGuebrjI+kUAH"
    "QMmDkbaTHWa4keyvN4TyNiFsGvLrvCBEUduddbj2aHkNHcA0/F3htmg4aL2ncSPNshbC0IXM"
    "gSCnSqZzOiSyFpFDx1mYWSlnoAMUXS8iPqHOtni7mFWzznrV77RfOFJlhtIduJ5bc9BeZXH5"
    "C7NOwVKKh4LqnIo7MhV3BOB1zeEqM1DKANiw0950VjTlybWX9/IzlRMszKygiYpOxp2lnXo5"
    "Hb/u20Sh9jL/SV5FJ+8Ulip1HUDBfC615u/TPVONBQYqIVtr7FU+gLZhzfucK2aHoq0/6lfX"
    "jpkDvPv0Cudvpp21fYnPAk2u5g5qImnRffP3mze/pVKoWAz3X4DfyqHWCugH9+YAAAAASUVO"
    "RK5CYII="
)


"""Toolbar icon for refresh button."""
ToolbarRefresh = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAABQAAAAUCAYAAACNiR0NAAABnUlEQVQ4y9XUPWtUQRTG8d/c"
    "rGuljeJLIwmYtTeKinZ2YuHdBUmnYCUWhuRjKDZaxiaVKO4mEPIBFFGxymKxIQiCaJqAik0I"
    "ZsciN5u7d2/WlHq6M+fMf+Y88zD86xGGVp+pqJrAOEbwWdV7122AlpOia+qeDgfOq+qawZTg"
    "WKH6SzSLWUELidT43sCXjgqWBOf3OeUnqdM7SdJXWnJAYrEHi9q4hZpoDHW8HUav9GUbZgQX"
    "MticTXdM+p2Tois6MQyY5JpHBFMZbHkAtr2+gLECI5bfsOuc4Him7KMB2HbcxJEC7mc5MKjl"
    "mt6UzpNaxSpoOpTpS0uU6hQ1/CC6mwHX9uHgU3icZZMUgamOlqrosOCsZm/j+s7phVEv90wX"
    "rZS/MguC0cLGd7jUt/bCQcF0Vv9qzXK5D2PB6FFHUB+AVczhTLbywL3dl678RakfuKKpLahk"
    "Y07nYK8lnuxt7H5vBcFFPO/dO/R1vBLd0LBVbuzdxo+imuhh0WNZfMF9m65q+D78+2q6LVrU"
    "sJ77dSYwii3Bim/aec3+v/gDVpZ0g1sFQpUAAAAASUVORK5CYII="
)


"""Toolbar icon for undo buttons."""
ToolbarUndo = PyEmbeddedImage(
    "iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABHNCSVQICAgIfAhkiAAAAhlJ"
    "REFUOI2Vk1FIU1Ecxn/37l6vM5dtmYkVZRLUHooiWJBBQRBFPsQeeohiQVC99iB7E6NoCEb0"
    "EoGBBT0Ejh4yw5dsaZTakzgRjUzQNVjtatPJrp5zehBrUxf2vZ3v8H38/9/5DhRBXzuq2F0+"
    "jNVEz2MqTYvURsQAev4hHMZrWqQCwehG9X8nCIfxnt9POhCMwmwYWF4jp/mSUuhJ5SqJdw8l"
    "rre2Mp9voK0WL6bvYOg2uIOglhBCoRC5D90vvpgaWs/w/JnmloWpFQNXUxNlZ/fxKxCMsvjz"
    "Npquk8sBYhInO4b9YxBU1qjzn66y3K5Kr56+uPvA0rtYr/wO4Lp7FWdFrOuuZVfDRCmFlAIp"
    "JVLMkcuOYZVZ2qaKnZs9MnOk9lD2eSzGot45iq8/GsT0nANACIeJr0NMfoszPT1eEFjGjuOr"
    "9lPqNg+e2LMttjaDhkYmRu5T67+FFJkCscy0kUyZ5BzB1pobDH/smD91ebrcAIhEsAnjg5Z0"
    "oKERgMGuR4Xv5dbwH7vJbPIpVqmDhjQgrweRCHbnKL7+Vy1/NPUhtPoQGorDczPqQvztw4Gq"
    "XVcQ0oMGVtFy9LWj1qtyXxs1nzp0NZu4J98/86qCCfJRH1rOZg1/jYQuytVCZgRp+GaKGhQz"
    "6X3C3i07jko7lQDUA1jnM/0LSud4xfa6rtGBjqWTl+zm/zagpLp8/PPrN9KxX65QvwG5L9sQ"
    "XyS9uAAAAABJRU5ErkJggg=="
)
